#include "vin.h"
#include "adc.h"
#include "usart2.h" // 确保包含 huart2 定义所在的头文件
#include "usart1.h" 
#include <string.h>

/**
 * @brief 读取 ADC 值并转换为电压
 * @return float 电压值 (0.0V - 3.3V)
 */
float Get_ADC_Voltage(void)
{
    uint32_t adc_value = 0;
    float voltage = 0.0;

    // 1. 启动 ADC 转换
    HAL_ADC_Start(&hadc1);

    // 2. 等待转换完成 (超时时间设为 10ms)
    if (HAL_ADC_PollForConversion(&hadc1, 10) == HAL_OK)
    {
        // 3. 读取 12 位的数字值 (0 - 4095)
        adc_value = HAL_ADC_GetValue(&hadc1);

        // 4. 转换为电压值
        // 公式：电压 = (读取值 / 最大值) * 参考电压
        // 12位 ADC 最大值是 2^12 - 1 = 4095
        // STM32 标准参考电压通常是 3.3V
        voltage = (float)adc_value * 3.3f / 4095.0f;
    }

    // 5. 停止 ADC (单次模式下可选，但建议养成习惯)
    HAL_ADC_Stop(&hadc1);

    return voltage;
}

float Get_Vin_Voltage(void)
{
    float voltage = 11.0f * Get_ADC_Voltage();
    return (float)((int)(voltage * 100.0f + 0.5f))/100.0f;
}


void USART_Send_Voltage(void)
{
    uint8_t frame[8];
    
    // 1. 将电压放大100倍转换为整数
    // +0.5f 是为了四舍五入，防止浮点数精度误差（如 12.34 变成 12.33999）
    uint32_t v_int = (uint32_t)(Get_Vin_Voltage() * 100.0f + 0.5f); 

    // 限制最大值，防止输入超出 99.99 (可选)
    if (v_int > 999999) v_int = 999999; // 3字节理论最大值远超99.99

    frame[0] = 0x01;          // 帧头
    frame[1] = 0xFF;          // mode 固定为 FF
    frame[2] = 0xFF;          // value 固定为 FF
    
    // 2. 将数据填入 3 个预留位 (大端模式：高位在前)
    // 例如 v_int = 9999 (0x00270F)
    frame[3] = 0x01; // 高位 (0x00)
    frame[4] = (uint8_t)((v_int >> 8) & 0xFF);  // 中位 (0x27)
    frame[5] = (uint8_t)(v_int & 0xFF);         // 低位 (0x0F)
    
    // 3. 生成和校验 (Sum Check)
    // 校验位 = frame[1] + frame[2] + frame[3] + frame[4] + frame[5]
    uint8_t checksum = 0;
    for(int i = 1; i <= 5; i++)
    {
        checksum += frame[i];
    }
    frame[6] = checksum;
    
    frame[7] = 0xFF;          // 停止位
    
    // 4. 阻塞式发送 8 字节二进制数据
    HAL_UART_Transmit(&huart2, frame, 8, 100);
}
