#include "AudioManager.h"
#include "PubSubClient.h" // MQTTManager 内部使用


// 构造函数实现
AudioManager::AudioManager(int bckPin, int lrcPin, int dinPin, int sampleRate, int dmaBufCount, int dmaBufLen)
: _bckPin(bckPin), _lrcPin(lrcPin), _dinPin(dinPin),
  _sampleRate(sampleRate), _dmaBufCount(dmaBufCount), _dmaBufLen(dmaBufLen) {
    // 构造函数体
    
}

// 初始化 I2S 实现
bool AudioManager::begin() {
    Serial.println("Initializing I2S...");

    i2s_config_t i2s_config = {
        .mode = (i2s_mode_t)(I2S_MODE_MASTER | I2S_MODE_TX), // 设置为主机模式，发送数据
        .sample_rate = _sampleRate,                          // 设置采样率
        .bits_per_sample = (i2s_bits_per_sample_t)_bitsPerSample, // 设置每个采样点的位数
        .channel_format = _channelFormat,                    // 通道格式
        .communication_format = _commFormat,                 // 通信格式
        .intr_alloc_flags = ESP_INTR_FLAG_LEVEL1,            // 中断分配标志
        .dma_buf_count = _dmaBufCount,                      // DMA缓冲区数量
        .dma_buf_len = _dmaBufLen,                          // 每个DMA缓冲区的长度
        .use_apll = false,                                   // 不使用APLL
        .tx_desc_auto_clear = true,                          // 自动清除已发送的TX描述符
        .fixed_mclk = 0                                      // 不使用固定的MCLK
    };

    // 安装I2S驱动
    esp_err_t err = i2s_driver_install(_i2sPort, &i2s_config, 0, NULL);
    if (err != ESP_OK) {
        Serial.printf("I2S driver install failed: %d\n", err);
        return false;
    }
    Serial.println("I2S driver installed.");

    // 配置I2S引脚
    i2s_pin_config_t pin_config = {
        .bck_io_num = _bckPin,         // 位时钟引脚
        .ws_io_num = _lrcPin,           // 左右时钟/字选择引脚
        .data_out_num = _dinPin,        // 数据输出引脚
        .data_in_num = I2S_PIN_NO_CHANGE // 不使用数据输入引脚
    };

    err = i2s_set_pin(_i2sPort, &pin_config);
    if (err != ESP_OK) {
        Serial.printf("I2S pin setting failed: %d\n", err);
        return false;
    }
    Serial.println("I2S pins set.");

    return true;
}

// 请求爆炸声音播放实现
void AudioManager::requestPlay() {
    _audioplay_index =false;
    _playRequested = true;
}

// 请求滴滴声音声音播放实现
void AudioManager::requestPlay_Drip() {
    _audioplay_index =true;
    _playRequested = true;
}

// 检查是否正在播放（简单实现）
bool AudioManager::isPlaying() const {
    // 更精确的实现需要跟踪播放状态，这里只是基于请求标志
    return _playRequested; // 在play()函数结束时需要将此标志设回false
}

// 内部播放函数实现,暂定播放滴滴声
void AudioManager::play_Drip_sound() {
    Serial.println("Start playing_Drip_sound audio...");
    size_t bytes_written = 0;
    int current_pos = 0;

    // 创建一个RAM缓冲区，用于从PROGMEM拷贝数据块
    uint8_t ram_buffer[_dmaBufLen];

    // 循环直到所有音频数据都被发送
    while (current_pos < pcm_sound_data_len_Drip) {
        int bytes_to_copy = _dmaBufLen;
        if (current_pos + bytes_to_copy > pcm_sound_data_len_Drip) {
            bytes_to_copy = pcm_sound_data_len_Drip - current_pos;
        }

        // 从PROGMEM拷贝数据到RAM缓冲区
        memcpy_P(ram_buffer, &pcm_sound_data_Drip[current_pos], bytes_to_copy);

        // 将RAM缓冲区中的数据写入I2S总线
        esp_err_t result = i2s_write(_i2sPort, ram_buffer, bytes_to_copy, &bytes_written, portMAX_DELAY);

        if (result != ESP_OK) {
            Serial.printf("I2S write error: %d\n", result);
            break;
        }
        // if (bytes_written < bytes_to_copy) {
        //     Serial.printf("I2S write incomplete: Wrote %d / %d bytes\n", bytes_written, bytes_to_copy);
        // }

        current_pos += bytes_written; // 更新当前位置

        
    }

    // 播放完毕后，清空DMA缓冲区确保静音
    i2s_zero_dma_buffer(_i2sPort);

    Serial.println("Audio playback finished.");

    _playRequested = false; // 重置播放请求标志

    // 播放结束后，如果 MQTTManager 指针有效，发送播放结束消息
    // if (_mqttManager) {
    //     Serial.println("Publishing audio play end message...");
    //     _mqttManager->publishJson("audioplayend", 1);
    // }
}


// 内部播放函数实现
void AudioManager::play() {
    Serial.println("Start playing audio...");
    size_t bytes_written = 0;
    int current_pos = 0;

    // 创建一个RAM缓冲区，用于从PROGMEM拷贝数据块
    uint8_t ram_buffer[_dmaBufLen];

    // 循环直到所有音频数据都被发送
    while (current_pos < pcm_sound_data_len) {
        int bytes_to_copy = _dmaBufLen;
        if (current_pos + bytes_to_copy > pcm_sound_data_len) {
            bytes_to_copy = pcm_sound_data_len - current_pos;
        }

        // 从PROGMEM拷贝数据到RAM缓冲区
        memcpy_P(ram_buffer, &pcm_sound_data[current_pos], bytes_to_copy);

        // 将RAM缓冲区中的数据写入I2S总线
        esp_err_t result = i2s_write(_i2sPort, ram_buffer, bytes_to_copy, &bytes_written, portMAX_DELAY);

        if (result != ESP_OK) {
            Serial.printf("I2S write error: %d\n", result);
            break;
        }
        // if (bytes_written < bytes_to_copy) {
        //     Serial.printf("I2S write incomplete: Wrote %d / %d bytes\n", bytes_written, bytes_to_copy);
        // }

        current_pos += bytes_written; // 更新当前位置
    }

    // 播放完毕后，清空DMA缓冲区确保静音
    i2s_zero_dma_buffer(_i2sPort);

    Serial.println("Audio playback finished.");

    _playRequested = false; // 重置播放请求标志

    // 播放结束后，如果 MQTTManager 指针有效，发送播放结束消息
    // if (_mqttManager) {
    //     Serial.println("Publishing audio play end message...");
    //     _mqttManager->publishJson("audioplayend", 1);
    // }
}

// FreeRTOS 任务函数实现
// 这个函数运行在一个单独的核心上，负责检查播放请求并调用播放函数
void audioPlayTask(void* pvParameters) {
    AudioManager* audioManager = static_cast<AudioManager*>(pvParameters);
    if (!audioManager) {
        Serial.println("AudioPlay Task: Invalid parameter!");
        vTaskDelete(NULL); // 删除当前任务
        return;
    }

    for (;;) {
        if (audioManager->_playRequested) {
            if(audioManager->_audioplay_index == false) audioManager->play();
            if(audioManager->_audioplay_index == true ) audioManager->play_Drip_sound();
        }
        vTaskDelay(pdMS_TO_TICKS(50)); // 短暂延时，避免忙等
    }
}