#ifndef MQTT_MANAGER_H
#define MQTT_MANAGER_H

#include <Arduino.h>
#include <WiFiClient.h>
#include <PubSubClient.h>
#include <ArduinoJson.h> // 需要用于发布JSON
#include <functional>    // For std::function
#include <string>        // For std::string
#include "DeviceControl.h"

// 定义 MQTT 回调函数类型
using MqttCallbackType = std::function<void(char* topic, byte* payload, unsigned int length)>;

class DeviceControl;

class MQTTManager {
private:
    WiFiClient _espClient;
    PubSubClient _mqttClient; // 必须在构造函数初始化列表中初始化

    std::string _broker;
    int _port;
    std::string _username;
    std::string _password;
    std::string _clientId; // 使用 std::string 以便动态生成
    std::string _subscribeTopic;
    std::string _publishTopic;

    DeviceControl *_DeviceControl = nullptr;
    

    MqttCallbackType _callback; // 存储用户提供的回调函数

    unsigned long lastReconnectAttempt = 0;

    // PubSubClient 的回调函数需要静态或全局
    // static void staticMqttCallback(char* topic, byte* payload, unsigned int length); // 备选方案
    // 使用类的 friend 任务函数或一个公共的静态方法来桥接 PubSubClient 回调

    // 内部连接尝试函数
    bool tryConnect();

public:
    // 构造函数
    // 传入 MQTT Broker 地址、端口、用户名、密码和话题
    MQTTManager(const char* broker, int port, const char* username, const char* password,
                const char* subscribeTopic, const char* publishTopic);

    // 初始化方法
    void begin();

    // 设置 DeviceControl 指针
    void setDeviceControl( DeviceControl *deviceControl) {
        _DeviceControl = deviceControl; 
        }

    // 设置消息回调函数
    void setCallback(MqttCallbackType callback);

    // 在 FreeRTOS 任务中调用，保持 MQTT 连接并处理消息
    void loop();

    // 发布消息 (纯文本)
    bool publish(const char* topic, const char* payload);

    // 发布 JSON 消息
    bool publishJson(const char* action, int value);

    // 检查 MQTT 连接状态
    bool isConnected() { return _mqttClient.connected(); }

    // 提供一个静态函数，作为 PubSubClient 的回调入口
    // 这个函数会将调用转发到当前活跃的 MQTTManager 实例的实际回调函数
    static void pubSubCallbackWrapper(char* topic, byte* payload, unsigned int length);

    // 需要一个指向当前活跃实例的静态指针，以便回调 wrapper 可以访问
    static MQTTManager* _instance;

    // FreeRTOS 任务函数（需要在主文件或单独的.cpp中实现，并友元声明）
    friend void mqttLoopTask(void* pvParameters);

    int wifi_connected_is();
};

// FreeRTOS 任务函数声明
void mqttLoopTask(void* pvParameters);

extern MQTTManager mqttManager;

#endif // MQTT_MANAGER_H