#include "DeviceControl.h"
#include "AudioManager.h"
#include <ArduinoJson.h> // 需要用于 handleMQTTCommand


// 构造函数实现
DeviceControl::DeviceControl(int pin1, int pin2, int pin3, int pin4, int sensorPin)
: _sensorPin(sensorPin) {
    _outputPins[0] = pin1;
    _outputPins[1] = pin2;
    _outputPins[2] = pin3;
    _outputPins[3] = pin4;
}

// begin 方法实现
void DeviceControl::begin() {
    Serial.println("DeviceControl initializing...");
    // 配置引脚模式
    for (int i = 0; i < 4; ++i) {
        pinMode(_outputPins[i], OUTPUT);
        digitalWrite(_outputPins[i], LOW); // 确保启动时为低电平
    }
    pinMode(_sensorPin, INPUT); // 传感器引脚设为输入
    Serial.println("DeviceControl initialized.");
}

// 设置数字输出实现 (1-4 对应 _outputPins 数组索引 0-3)
void DeviceControl::setPinValue(int pinIndex, int val) {
    if (pinIndex >= 1 && pinIndex <= 4) {
        //Serial.printf("Setting digital pin %d (%d) to %d\n", pinIndex, _outputPins[pinIndex - 1], val);
        digitalWrite(_outputPins[pinIndex - 1], val == 1 ? HIGH : LOW);
    } else {
        Serial.printf("Warning: Invalid pin index for digital write: %d\n", pinIndex);
    }
}

// 设置 PWM 输出实现 (1-4 对应 _outputPins 数组索引 0-3)
void DeviceControl::setPWMValue(int pinIndex, int val) {
     if (pinIndex >= 1 && pinIndex <= 4) {
        // analogWrite 在 ESP32 上是 8 位分辨率 (0-255)
        int analogVal = constrain(val * 255 / 100, 0, 255); // 将输入的 0-100 值映射到 0-255
        //Serial.printf("Setting PWM pin %d (%d) to %d (scaled to %d)\n", pinIndex, _outputPins[pinIndex - 1], val, analogVal);
        analogWrite(_outputPins[pinIndex - 1], analogVal);
    } else {
        Serial.printf("Warning: Invalid pin index for PWM write: %d\n", pinIndex);
    }
}

// 更新 LED 血量效果实现
void DeviceControl::updateLedBlood() {
  // 这个函数需要在周期性的任务中被调用，比如每几十毫秒一次
  _led_count++; // 每次调用都递增计数器
  if (_led_count > 20) _led_count = 0; // 闪烁周期计数器

  if (_ledblood >= 6) { // 满血或更高，3个灯全亮
    setPinValue(1,1);
    setPinValue(2,1);
    setPinValue(3,1);
  } else if (_ledblood == 5){ // 5格血，灯1闪烁，灯2、3常亮
    setPinValue(1, (_led_count < 10)); // 计数器 < 10 时为高电平
    setPinValue(2,1);
    setPinValue(3,1);
  } else if (_ledblood == 4){ // 4格血，灯1灭，灯2、3常亮
    setPinValue(1,0);
    setPinValue(2,1);
    setPinValue(3,1);
  } else if (_ledblood == 3){ // 3格血，灯2闪烁，灯3常亮，灯1灭
    setPinValue(2, (_led_count < 10));
    setPinValue(1,0);
    setPinValue(3,1);
  } else if (_ledblood == 2){ // 2格血，灯3常亮，灯1、2灭
    setPinValue(1,0);
    setPinValue(2,0);
    setPinValue(3,1);
  } else if (_ledblood == 1){ // 1格血，灯3闪烁，灯1、2灭
    setPinValue(3, (_led_count < 10));
    setPinValue(1,0);
    setPinValue(2,0);
  } else { // ledblood <= 0，所有灯灭
    setPinValue(1,0);
    setPinValue(2,0);
    setPinValue(3,0);
  }
}
 


// 读取传感器值实现
int DeviceControl::readSensor() {
    // int rawValue = analogRead(_sensorPin);
    // // 将读数转换为电压（假设 ADC 12位，参考电压 3.3V 或 2.5V）
    // // ESP32C3 ADC 可以是 12位 (0-4095)，参考电压通常是 3.3V 内部或外部。
    // // 原始代码使用了 2.5，我们假设是 2.5V 参考电压，或者原始代码有误。
    // // 使用 3.3V 参考电压是更常见的做法: rawValue * 3.3 / 4095.0
    // // 如果你的电路使用了外部 2.5V 参考或其他方式，请调整下面的乘数
    // const double referenceVoltage = 3.3; // 假设使用 3.3V 参考电压
    // double voltage = (double)rawValue * referenceVoltage / 4095.0;
    // Serial.printf("Sensor raw: %d, Voltage: %.2fV\n", rawValue, voltage); // 调试信息
    int digitalValue = digitalRead(_sensorPin);
    return (digitalValue == HIGH) ? 1 : 0;
    // return voltage;
}

// 处理来自 MQTT 的控制指令
void DeviceControl::handleMQTTCommand(int message_Type, JsonObject body) {

      message_type msgType = static_cast<message_type>(message_Type); ;

    if (msgType == message_type::PIN_CONTROL || msgType == message_type::PWM_CONTROL) { // GPIO 或 PWM 控制
        int pin = 0, val = 0;
        if (body.containsKey("pin")) {
            pin = body["pin"]; // 获取引脚编号 (1-4)
        }
        if (body.containsKey("val")) {
            val = body["val"]; // 获取值
        }

        if (msgType == message_type::PIN_CONTROL) { // 数字IO控制
            setPinValue(pin, val);
        } else if (msgType == message_type::PWM_CONTROL) { // PWM控制
            setPWMValue(pin, val);
        }
            Serial.printf("Processed command: pin=%d, val=%d (Type %d)\n", pin, val, msgType);
        } else if (msgType == message_type::LED_CONTROL||msgType == message_type::AUDIO_CONTROL||msgType == message_type::SMOKE_CONTROL
                    ||msgType == message_type::LED_INIT_CONTROL) { // 动作控制
            
            if (body.containsKey("action")) {
                String action = body["action"].as<String>();
                int val = 0;
                if (body.containsKey("val")) {
                    val = body["val"]; // 获取值
                }

                if (action == "led") {
                    setLedBlood(val); // 设置 LED 血量Connected
                    Serial.printf("Processed action: led, val=%d\n", val);
                } else if (action == "audio") {// audio 动作在 MQTT 回调中处理并请求 AudioManager 播放
                    audioManager.requestPlay(); // 请求音频播放
                    _mqttManager->publishJson("audioplayend", 1);
                    // 原代码中有 time_count=0，这里可以根据需要实现烟雾机控制逻辑
                    Serial.printf("audio playend\n", val);
                }else if (action == "smoke_ms") {
                    if(body.containsKey("open")) int open = body["open"]; 
                    _smoke_count=0;
                    _smoke_ms=val;
                    _smoke_index=true;
                    if(open == 0) {
                        setPinValue(4,0); 
                        _smoke_index =false;
                    }
                    // 原代码中有 time_count=0，这里可以根据需要实现烟雾机控制逻辑
                    Serial.printf("smoke set\n",val);
                }else if(action =="led_init"){
                    setLedBlood(val);
                    Serial.printf("smoke set\n",val);
                }else {
                    Serial.printf("Unknown action: %s\n", action.c_str());
                }
        }
    }
}

void DeviceControl::updatecontrol_smoke() {

    if(_smoke_index = true){
        int smoke_i = _smoke_ms / 20;
        if(_smoke_count < smoke_i){
            setPinValue(4,1); 
        }else setPinValue(4,0); 
         _smoke_count++;
        if(_smoke_count>60000) _smoke_count =60001;
    }

}

void DeviceControl::updatebeartheat(){
    _beat_count++;
    if(_beat_count>60) {
        _beat_count=0;
        Serial.print("心跳发送");
        _mqttManager->publishJson("heartbeat", 1);
    }if(_beat_count>100) _beat_count=101;
}

// 更新方法，在周期性任务中调用
void DeviceControl::update() {

    // 更新 LED 状态
    updateLedBlood();
    // 更新 烟雾 状态
    updatecontrol_smoke();

    //更新心跳
    updatebeartheat();
    // 读取传感器并发布数据 (可以设置一个频率，而不是每次 update() 都发布)
    // 简单的实现：每隔一定时间或阈值触发时发布
    // static unsigned long lastSensorPublishTime = 0;
    // const unsigned long sensorPublishInterval = 50; // 每 1 秒发布一次传感器数据
    // _shot_count++;
    // if(_shot_count < 6){
    //     return ;
    // }else {
    //     _shot_count=7;
    // }
    // if (millis() - lastSensorPublishTime >= sensorPublishInterval) {
         //double sensorVoltage = readSensor();

         // 原始代码中的阈值判断
         if (readSensor() == 1&& mqttManager.wifi_connected_is()==0) { // 传感器值超过阈值
             if (_mqttManager) {
                // 发布 "shoted" 消息
                _mqttManager->publishJson("shoted", 1); 
                audioManager.requestPlay();
               // Serial.print(sensorVoltage);
             } else {
                Serial.println("MQTTManager not set, cannot publish sensor data.");
             }
             //_shot_count=0;
         }
    //      lastSensorPublishTime = millis();
    // }
     
    
}


// FreeRTOS 任务函数实现
void devicePeriodicTask(void* pvParameters) {
    DeviceControl* deviceControl = static_cast<DeviceControl*>(pvParameters);
     if (!deviceControl) {
        Serial.println("Device Periodic Task: Invalid parameter!");
        vTaskDelete(NULL);
        return;
    }

    for (;;) {
        deviceControl->update(); // 调用 DeviceControl 的 update 方法
        vTaskDelay(pdMS_TO_TICKS(20)); // 每 50 毫秒更新一次
    }
}