#include "MQTTManager.h"
#include <WiFi.h> // 需要检查 WiFi 连接状态
#include <string> // For std::string

// 初始化静态成员指针
MQTTManager* MQTTManager::_instance = nullptr;

// 构造函数实现
MQTTManager::MQTTManager(const char* broker, int port, const char* username, const char* password,
                         const char* subscribeTopic, const char* publishTopic)
: _mqttClient(_espClient), // 初始化 PubSubClient
  _broker(broker), _port(port), _username(username), _password(password),
  _subscribeTopic(subscribeTopic), _publishTopic(publishTopic) {

    // 在构造函数中设置静态实例指针
    _instance = this;
}

// begin 方法实现
void MQTTManager::begin() {
    Serial.println("MQTTManager initializing...");
    
    _mqttClient.setServer(_broker.c_str(), _port);
    // 设置 PubSubClient 的回调函数为我们的 wrapper
    _mqttClient.setCallback(MQTTManager::pubSubCallbackWrapper);

    // 生成 MQTT 客户端ID
    uint8_t mac[6];
    WiFi.macAddress(mac);
    char clientIdChar[20]; // 需要足够的空间
    sprintf(clientIdChar, "ESP32-%02X%02X%02X%02X%02X%02X", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
    _clientId = clientIdChar;

    Serial.print("Generated MQTT Client ID: ");
    Serial.println(_clientId.c_str());
}

// 设置消息回调函数
void MQTTManager::setCallback(MqttCallbackType callback) {
    _callback = callback;
}

// PubSubClient 回调包装函数实现
// 这是 PubSubClient 调用的函数，它再调用我们存储的用户回调
void MQTTManager::pubSubCallbackWrapper(char* topic, byte* payload, unsigned int length) {
    if (_instance && _instance->_callback) {
        // 调用存储的用户回调函数
        _instance->_callback(topic, payload, length);
    }
}


// MQTT 连接尝试实现
bool MQTTManager::tryConnect() {
    Serial.print("Attempting MQTT connection...");
    // 使用客户端 ID、用户名和密码进行连接
    if (_mqttClient.connect(_clientId.c_str(), _username.c_str(), _password.c_str())) {
        Serial.println("connected!");
        // 订阅主题
        _mqttClient.subscribe(_subscribeTopic.c_str());
        Serial.print("Subscribed to topic: ");
        Serial.println(_subscribeTopic.c_str());
        return true;
    } else {
        Serial.print("failed, rc=");
        Serial.print(_mqttClient.state());
        Serial.println(" try again in 5 seconds");
        return false;
    }
}


// loop 方法实现 (由任务调用)
void MQTTManager::loop() {
    if (WiFi.status() == WL_CONNECTED) { // 只有在 WiFi 连接时才处理 MQTT
      if (!_mqttClient.connected()) { // 如果 MQTT 未连接

      unsigned long now = millis();
      if (now - lastReconnectAttempt > 5000) {
                lastReconnectAttempt = now;
                if (tryConnect()) {
                    lastReconnectAttempt = 0;
                }
      }
      } else {
        _mqttClient.loop(); // 保持 MQTT 连接和处理消息
      }
    }
     // 如果 WiFi 未连接，等待 WiFi 连接的任务处理
}

// 发布消息 (纯文本) 实现
bool MQTTManager::publish(const char* topic, const char* payload) {
    if (_mqttClient.connected()) {
        return _mqttClient.publish(topic, payload);
    }
    Serial.println("MQTT not connected, cannot publish.");
    return false;
}

// 发布 JSON 消息实现
bool MQTTManager::publishJson(const char* action, int value) {
     if (!_mqttClient.connected()) {
        Serial.println("MQTT not connected, cannot publish JSON.");
        return false;
    }

    // 创建JSON对象
    DynamicJsonDocument doc(256); // 根据你的JSON文档大小调整

    
    // 构建head部分
    JsonObject head = doc.createNestedObject("head");
    // 构建body部分
    JsonObject body = doc.createNestedObject("body");

    if(action == "audioplayend"){
        head["message_type"] = 3102; // 根据你的协议调整
        body["action"] = action; // 传入 action 字符串
        body["val"] = value;   // 传入 val 值
 
    }else if(action == "shoted"){
        head["message_type"] = 3101; // 根据你的协议调整 
        body["action"] = action; // 传入 action 字符串
        body["val"] = value;   // 传入 val 值
    }else if(action =="heartbeat") {
        head["message_type"] = 1;
        body["val"] = value;
         // 2. 添加空指针检查
        if (_DeviceControl != nullptr) {
       
            body["led"] = _DeviceControl->getLedBlood();
            body["smokeindex"] = _DeviceControl->getsmoke();
        } else {
            // 处理指针为空的情况，避免崩溃
            body["led"] = 0;  // 默认值
            body["smokeindex"] = false;  // 默认值
            Serial.println("Error: DeviceControl pointer is null!");
        }
    }
    
   

    // 序列化JSON为字符串
    String jsonString;
    serializeJson(doc, jsonString);

    Serial.print("Publishing JSON message to ");
    Serial.print(_publishTopic.c_str());
    Serial.print(": ");
    Serial.println(jsonString);

    // 发布消息
    return _mqttClient.publish(_publishTopic.c_str(), jsonString.c_str());
}

// FreeRTOS 任务函数实现
void mqttLoopTask(void* pvParameters) {
    MQTTManager* mqttManager = static_cast<MQTTManager*>(pvParameters);
    if (!mqttManager) {
        Serial.println("MQTT Task: Invalid parameter!");
        vTaskDelete(NULL);
        return;
    }

    for (;;) {
        mqttManager->loop(); // 调用 MQTTManager 的 loop 方法
        vTaskDelay(pdMS_TO_TICKS(100)); // 短暂延时
    }
}

int MQTTManager::wifi_connected_is() {
    return _mqttClient.state();
}