#include "WiFiConfigManager.h"
#include <functional> // Needed for std::bind

// 构造函数实现
WiFiConfigManager::WiFiConfigManager(int port) : _server(port) {
    // Preferences 对象不需要在这里初始化，在 begin() 或 load/save 中初始化
}

// begin 方法实现
bool WiFiConfigManager::begin() {
    Serial.println("WiFiConfigManager initializing...");

    // 加载保存的 WiFi 凭据
    loadCredentials();

    // 尝试连接 WiFi (如果凭据有效)
    bool connected = false;
    if (_ssid.length() > 0) {
        Serial.print("Attempting to connect to saved WiFi: ");
        Serial.println(_ssid);

        WiFi.mode(WIFI_STA); // 设置为站点模式
        WiFi.begin(_ssid.c_str(), _password.c_str());

        // 等待连接或超时
        int timeout_sec = 15; // 15秒超时
        while (timeout_sec > 0 && WiFi.status() != WL_CONNECTED) {
            delay(1000);
            Serial.print(".");
            timeout_sec--;
        }
        Serial.println(); // 换行

        if (WiFi.status() == WL_CONNECTED) {
            Serial.println("WiFi connected successfully!");
            Serial.print("IP Address: ");
            Serial.println(WiFi.localIP());

            connected = true;
        } else {
            Serial.println("WiFi connection failed.");
            // 连接失败，清除凭据并进入 AP 模式
            clearCredentials();
            _ssid = ""; _password = ""; // 清除内存中的凭据
        }
    }

    // 如果没有连接成功，启动 AP 配网模式
    if (!connected) {
        startConfigPortal();
    } else {
        // 如果连接成功，为了方便用户随时重新配置，也可以启动 WebServer
        // 但是此时 WebServer 运行在 Station 模式下，需要通过分配的 IP 访问
        //startConfigPortal(); // 启动 WebServer，但 WiFi 保持 STA 模式

         if (WiFi.getMode() == WIFI_STA && WiFi.isConnected()) {
            _server.on("/", std::bind(&WiFiConfigManager::handleRoot, this));
            _server.on("/save", HTTP_POST, std::bind(&WiFiConfigManager::handleSave, this));
            // _server.onNotFound(std::bind(&WiFiConfigManager::handleNotFound, this)); // 如果需要
            // 启动 Web 服务器 (如果尚未启动)
            _server.begin();
             Serial.print("Web config available at http://");
             Serial.println(WiFi.localIP());
         }
    }

    return connected;
}

// loop 方法实现
void WiFiConfigManager::loop() {
    _server.handleClient(); // 处理 WebServer 的客户端请求
}

// 启动 AP 配网模式实现
void WiFiConfigManager::startConfigPortal() {
    Serial.println("Starting AP config portal...");

    // 如果当前不是 AP 模式，切换到 AP 模式
    if(WiFi.getMode() != WIFI_AP) {
       WiFi.mode(WIFI_AP); // 设置 WiFi 为 AP 模式
       // 启动一个名为 "ESP32C3_Config"，密码为 "12345678" 的 WiFi 热点
       WiFi.softAP("ESP32C3_Config", "12345678");
       delay(100); // 给 AP 一点时间启动
       IPAddress apIP = WiFi.softAPIP(); // 获取 AP 的 IP 地址
       Serial.println("AP started!");
       Serial.println("Connect to SSID: ESP32C3_Config (Password: 12345678)");
       Serial.print("Open browser and visit: http://");
       Serial.println(apIP);
    }


    // 设置 Web 服务器的路由
    // 使用 std::bind 将成员函数绑定到 WebServer 回调
    _server.on("/", std::bind(&WiFiConfigManager::handleRoot, this));
    _server.on("/save", HTTP_POST, std::bind(&WiFiConfigManager::handleSave, this));
    // _server.onNotFound(std::bind(&WiFiConfigManager::handleNotFound, this)); // 如果需要

    // 启动 Web 服务器 (如果尚未启动)
    _server.begin();
    Serial.println("WebServer started.");
}

// 加载 WiFi 凭据实现
bool WiFiConfigManager::loadCredentials() {
    Serial.println("Loading WiFi credentials from Preferences...");
    _preferences.begin("wifi_config", true); // 打开 "wifi_config" 命名空间，true 表示只读模式
    _ssid = _preferences.getString("ssid", "");       // 读取 "ssid"
    _password = _preferences.getString("password", ""); // 读取 "password"
    _preferences.end(); // 关闭 Preferences

    Serial.print("Loaded SSID: '"); Serial.print(_ssid); Serial.println("'");
    Serial.print("Loaded Password: '"); Serial.print(_password); Serial.println("'");

    return _ssid.length() > 0; // 如果 SSID 不为空，认为加载成功
}

// 保存 WiFi 凭据实现
void WiFiConfigManager::saveCredentials() {
    Serial.println("Saving WiFi credentials to Preferences...");
    _preferences.begin("wifi_config", false); // 打开 "wifi_config" 命名空间，false 表示读写模式
    _preferences.putString("ssid", _ssid);        // 将 ssid 存入 Flash
    _preferences.putString("password", _password);  // 将 password 存入 Flash
    _preferences.end();                        // 关闭 Preferences

    Serial.println("Credentials saved.");
}

// 清除 WiFi 凭据实现
void WiFiConfigManager::clearCredentials() {
    Serial.println("Clearing WiFi credentials from Preferences...");
    _preferences.begin("wifi_config", false);
    _preferences.clear(); // 清除 "wifi_config" 命名空间下的所有数据
    _preferences.end();
    Serial.println("Credentials cleared.");
}


// WebServer 处理函数实现 (根目录)
void WiFiConfigManager::handleRoot() {
    Serial.println("Handling / request");
    _server.send(200, "text/html; charset=utf-8", _configHtml);
}

// WebServer 处理函数实现 (保存)
void WiFiConfigManager::handleSave() {
    Serial.println("Handling /save request...");
    if (_server.hasArg("ssid")) {
        _ssid = _server.arg("ssid");
        _password = _server.arg("password"); // password 可能为空

        saveCredentials(); // 保存到 Flash

        Serial.println("WiFi config saved. Restarting...");
        // 向客户端发送成功信息
        _server.send(200, "text/html; charset=utf-8", "<h3>保存成功，设备重启中...</h3><p>请等待设备重启并连接新的WiFi网络。</p>");
        delay(2000); // 延迟2秒，确保浏览器收到响应
        ESP.restart(); // 重启 ESP32
    } else {
        Serial.println("Error: Missing SSID argument in /save request.");
        _server.send(400, "text/plain; charset=utf-8", "错误：缺少 SSID 参数！");
    }
}

// 如果需要，可以实现 handleNotFound
// void WiFiConfigManager::handleNotFound() {
//     Serial.println("Handling Not Found");
//     _server.send(404, "text/plain", "Not found");
// }