// 引入核心库
#include <Arduino.h>
#include <freertos/FreeRTOS.h> // FreeRTOS 实时操作系统核心库
#include <freertos/task.h>     // FreeRTOS 任务管理库

// 引入自定义类头文件
#include "WiFiConfigManager.h"
#include "MQTTManager.h"
#include "AudioManager.h"
#include "DeviceControl.h"

// --- 按键配置 ---
const int BUTTON_PIN = D10;  // D10 引脚在 ESP32C3 上对应 GPIO10

// 按键检测常量
const unsigned long LONG_PRESS_DURATION_MS = 5000;  // 长按触发时间：5秒
const unsigned long DEBOUNCE_DELAY_MS = 50;         // 按键去抖动时间：50毫秒

// --- 模拟读入配置 ---
const int sensorPin0 = D0; // 模拟输入引脚 D0，在 ESP32-C3 上通常是 GPIO0

// -- 输出引脚 --
const int OUTPUT_PIN1 = D1; // 数字输出引脚 D1 -> GPIO1
const int OUTPUT_PIN2 = D2; // 数字输出引脚 D2 -> GPIO2
const int OUTPUT_PIN3 = D3; // 数字输出引脚 D3 -> GPIO3
const int OUTPUT_PIN4 = D4; // 数字输出引脚 D4 -> GPIO4

// -- I2S 引脚 --
const int I2S_BCLK = D5; // I2S 位时钟引脚
const int I2S_LRC = D6;  // I2S 左右时钟引脚
const int I2S_DIN = D7;  // I2S 数据输入引脚 (MAX98357A 不使用)

// --- MQTT 配置 ---
const char* MQTT_BROKER = "119.45.167.177"; // 你的 MQTT Broker IP 或 Hostname
const int MQTT_PORT = 1883;                     // MQTT 端口
const char* MQTT_USERNAME = "admin"; // 你的 MQTT 用户名
const char* MQTT_PASSWORD = "admin"; // 你的 MQTT 密码
const char* MQTT_SUBSCRIBE_TOPIC = "ser2dev/CN040200000001"; // 你要订阅的 MQTT 主题
const char* MQTT_RELEASE_TOPIC = "CN040200000001";   // 你要发布的 MQTT 主题

// --- 创建类实例 ---
// 这些实例现在包含了之前分散的全局变量和相关函数
WiFiConfigManager wifiManager; // 使用默认端口 80
MQTTManager mqttManager(MQTT_BROKER, MQTT_PORT, MQTT_USERNAME, MQTT_PASSWORD, MQTT_SUBSCRIBE_TOPIC, MQTT_RELEASE_TOPIC);
AudioManager audioManager(I2S_BCLK, I2S_LRC, I2S_DIN); // 使用默认采样率和缓冲区配置
DeviceControl deviceControl(OUTPUT_PIN1, OUTPUT_PIN2, OUTPUT_PIN3, OUTPUT_PIN4, sensorPin0);

// --- FreeRTOS 任务声明 ---
// 任务函数实现会放在 setup 或 loop 函数之后
void buttonMonitorTask(void* pvParameters); // 按键监控任务
// mqttLoopTask 在 MQTTManager.cpp 中声明和实现
// audioPlayTask 在 AudioManager.cpp 中声明和实现
// devicePeriodicTask 在 DeviceControl.cpp 中声明和实现


// --- MQTT 消息回调函数 ---
// 这个函数现在处理接收到的 MQTT 消息，并调用相应的类方法
void handleMqttMessage(char* topic, byte* payload, unsigned int length) {
  Serial.print("Received MQTT message - Topic: ");
  Serial.print(topic);
  Serial.print(", Payload: ");

  // 复制 payload 到一个以 null 结尾的字符串
  char messagePayload[length + 1];
  strncpy(messagePayload, (char*)payload, length);
  messagePayload[length] = '\0';
  Serial.println(messagePayload);

  // --- 解析 JSON 数据 ---
  StaticJsonDocument<256> doc; // 根据你的JSON文档大小调整

  DeserializationError error = deserializeJson(doc, messagePayload);

  if (error) {
    Serial.print("JSON parsing failed: ");
    Serial.println(error.c_str());
    return;
  }

  // 1. 解析 head 部分
  int message_type = 0;
  if (doc.containsKey("head")) {
    JsonObject head = doc["head"];
    if (head.containsKey("message_type")) {
      message_type = head["message_type"];
      Serial.print("  - Message Type (head.message_type): ");
      Serial.println(message_type);
    }
  }

  // 2. 解析 body 部分并分发给相应的管理器
  if (doc.containsKey("body")) {
    JsonObject body = doc["body"];

    // 交给 DeviceControl 处理 GPIO/PWM 和 LED 血量控制
    deviceControl.handleMQTTCommand(message_type, body);

    // 如果是 audio 动作，请求 AudioManager 播放
    // if (message_type == 5 && body.containsKey("action") && body["action"].as<String>() == "audio") {
    //      int val = body["val"] | 0; // Get value, default to 0 if not present
    //      if (val == 1) { // Assuming val=1 means request playback
    //         audioManager.requestPlay(); // 请求音频播放
    //      }
    //      Serial.printf("Processed action: audio, val=%d\n", val);
    // }
  }
}


void setup() {
  Serial.begin(115200); // 启动串口通信
  delay(1000); // 等待串口稳定

  Serial.println("\n--- System Starting ---");

  //mqtt初始化
  mqttManager.begin(); 

  // 设置各管理器之间的依赖关系 (通过设置指针)
  audioManager.setMQTTManager(&mqttManager);
  deviceControl.setMQTTManager(&mqttManager);
  mqttManager.setCallback(handleMqttMessage); // 设置 MQTT 回调函数
  mqttManager.setDeviceControl(&deviceControl);

  // 初始化设备控制（引脚模式等）
  deviceControl.begin();

  
  //按键初始化
  pinMode(BUTTON_PIN, INPUT_PULLUP); 

  // 初始化音频管理器 (I2S)
  audioManager.begin();

  // 初始化 WiFi 配置管理器 (尝试连接或启动 AP)
  bool wifiConnected = wifiManager.begin();

  // 创建 FreeRTOS 任务
  // 任务名称, 栈大小, 参数, 优先级, 任务句柄
  xTaskCreate(buttonMonitorTask, "ButtonMonitor", 2048, (void*)&wifiManager, 1, NULL); // 将 wifiManager 实例地址传递给按键任务
  xTaskCreate(mqttLoopTask, "MQTTLoop", 4096, (void*)&mqttManager, 1, NULL); // 将 mqttManager 实例地址传递给 MQTT 任务
  xTaskCreate(audioPlayTask, "AudioPlay", 4096, (void*)&audioManager, 1, NULL); // 将 audioManager 实例地址传递给音频任务
  xTaskCreate(devicePeriodicTask, "DevicePeriodic", 2048, (void*)&deviceControl, 1, NULL); // 将 deviceControl 实例地址传递给设备任务


  // 启动时播放一次声音（指示启动完成或AP模式启动）
  audioManager.requestPlay_Drip(); // 请求播放启动音

}

void loop() {
  // 主循环中主要处理 Web 服务器客户端请求
  // 其他周期性或阻塞性任务都在 FreeRTOS 任务中运行
  wifiManager.loop();

  // 短暂延时，避免主循环空转
  delay(10);
}


// --- FreeRTOS 任务函数实现 ---

// 按键监控任务实现
void buttonMonitorTask(void* pvParameters) {
  // 从参数中获取 WiFiConfigManager 实例指针
  WiFiConfigManager* wifiManager = static_cast<WiFiConfigManager*>(pvParameters);
   if (!wifiManager) {
        Serial.println("Button Monitor Task: Invalid WiFiManager parameter!");
        vTaskDelete(NULL); // 删除当前任务
        return;
    }

  int lastButtonState = HIGH;             // 上一次按键状态
  unsigned long buttonPressStartTime = 0; // 按键按下时的 millis() 时间
  bool resetTriggered = false;            // 标志，防止长按期间重复触发

  for (;;) { // 无限循环
    int currentButtonState = digitalRead(BUTTON_PIN);

    // --- 按键去抖动 ---
    if (currentButtonState != lastButtonState) {
      vTaskDelay(pdMS_TO_TICKS(DEBOUNCE_DELAY_MS)); // 等待去抖动时间
      currentButtonState = digitalRead(BUTTON_PIN); // 再次读取以确认状态
    }

    if (currentButtonState == LOW) { // 如果按键被按下
      if (lastButtonState == HIGH) { // 如果是刚按下的瞬间
        buttonPressStartTime = millis();
        resetTriggered = false; // Reset trigger flag on new press
        Serial.println("Button D10 pressed...");
      }

      // --- 检测长按 ---
      if (!resetTriggered && ((millis() - buttonPressStartTime) >= LONG_PRESS_DURATION_MS)) {
        Serial.println("Long press detected on D10! Clearing WiFi config...");

        // 听到声音表示长按成功
        audioManager.requestPlay_Drip(); // 请求播放声音

        // 清除 WiFi 配置并重启
        wifiManager->clearCredentials();

        Serial.println("WiFi Config cleared. Device restarting...");
        resetTriggered = true; // Mark as triggered
        vTaskDelay(pdMS_TO_TICKS(100)); // 短暂延时
        ESP.restart(); // 重启设备
      }
    } else { // 如果按键未被按下
      // 短按逻辑（如果需要）可以在这里实现：检查 lastButtonState == LOW 且 resetTriggered == false
      if (lastButtonState == LOW && !resetTriggered) {
           // Serial.println("Short press detected on D10."); // uncomment if you want short press logic
           // Add short press action here if needed
      }
      buttonPressStartTime = 0; // Reset timer
    }

    lastButtonState = currentButtonState; // 更新按键状态

    vTaskDelay(pdMS_TO_TICKS(50)); // Task delay
  }
}

// MQTTLoopTask, AudioPlayTask, DevicePeriodicTask 的实现已经分别放在各自的 .cpp 文件中