#ifndef WIFI_CONFIG_MANAGER_H
#define WIFI_CONFIG_MANAGER_H

#include <Arduino.h>
#include <WiFi.h>         // WiFi 功能库
#include <WebServer.h>    // Web 服务器库，用于配网
#include <Preferences.h>  // Flash 非易失性存储库，用于保存WiFi配置
#include <functional>     // 需要 std::bind 或 lambda

class WiFiConfigManager {
private:
    String _ssid = "";
    String _password = "";
    Preferences _preferences;
    WebServer _server; // 必须在构造函数初始化列表中初始化

    // HTML 配置页面内容（作为内部常量）
    const char* _configHtml = R"rawliteral(
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>ESP32-C3 WiFi 配网</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <style>
    body { font-family: Arial, sans-serif; margin: 20px; background-color: #f4f4f4; color: #333; }
    h2 { color: #0056b3; }
    form { background-color: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
    input[type=text], input[type=password] {
      width: calc(100% - 22px);
      padding: 10px;
      margin: 8px 0;
      display: inline-block;
      border: 1px solid #ccc;
      border-radius: 4px;
      box-sizing: border-box;
    }
    input[type=submit] {
      background-color: #4CAF50;
      color: white;
      padding: 12px 20px;
      border: none;
      border-radius: 4px;
      cursor: pointer;
      font-size: 16px;
      width: 100%;
    }
    input[type=submit]:hover {
      background-color: #45a049;
    }
    h3 { color: #333; }
    .reset-info { margin-top: 20px; font-size: 0.9em; color: #666; }
  </style>
</head>
  <body>
    <h2>ESP32-C3 WiFi 配网</h2>
    <form action="/save" method="POST">
      <label for="ssid">SSID:</label><br>
      <input type="text" id="ssid" name="ssid" required><br>
      <label for="password">密码:</label><br>
      <input type="password" id="password" name="password"><br>
      <input type="submit" value="保存并重启">
    </form>
    <p class="reset-info">提示：长按 D10 引脚 5 秒可清除 WiFi 配置并重启设备。</p>
  </body>
</html>
)rawliteral";


    // WebServer 处理函数需要绑定到类的实例
    void handleRoot();
    void handleSave();
    // void handleNotFound(); // 如果需要可以添加

    // 内部辅助函数
    bool loadCredentials(); // 从 Preferences 加载
    void saveCredentials(); // 保存到 Preferences

public:
    // 构造函数，指定 WebServer 监听端口
    WiFiConfigManager(int port = 80);

    // 初始化方法：加载配置，尝试连接，或启动 AP
    // 返回 true 如果最终成功连接 WiFi
    bool begin();

    // 在 loop() 中调用，处理 WebServer 客户端
    void loop();

    // 启动 AP 配网模式
    void startConfigPortal();

    // 清除 WiFi 配置
    void clearCredentials();

    // 获取当前连接的 SSID 和密码 (如果已连接或加载)
    String getSSID() const { return _ssid; }
    String getPassword() const { return _password; }

    // 检查 WiFi 连接状态
    bool isWiFiConnected() const { return WiFi.status() == WL_CONNECTED; }

    // 获取 WebServer 实例的引用，供需要额外路由的地方使用 (可选)
    // WebServer& getServer() { return _server; }
};

#endif // WIFI_CONFIG_MANAGER_H