#include "lcd.h"
#include "esp_lcd_panel_io.h"
#include "esp_lcd_panel_vendor.h"
#include "esp_lcd_panel_ops.h"
#include "driver/spi_master.h"
#include "driver/gpio.h"
#include "esp_log.h"
#include "esp_timer.h"
#include "esp_heap_caps.h"

static const char *TAG = "LCD_DEBUG";

static esp_lcd_panel_handle_t panel_handle = NULL;
static lv_disp_draw_buf_t draw_buf;
static lv_disp_drv_t disp_drv;
SemaphoreHandle_t lvgl_mux = NULL;

// 1. 刷新回调 - 增加日志输出，确认回调是否在跑
static void lvgl_flush_cb(lv_disp_drv_t *drv, const lv_area_t *area, lv_color_t *color_map) {
    esp_lcd_panel_handle_t panel = (esp_lcd_panel_handle_t)drv->user_data;
    
    // 这里的坐标打印非常重要，看看是不是在刷新 0,0 坐标
    // ESP_LOGD(TAG, "Flush: x(%d-%d), y(%d-%d)", area->x1, area->x2, area->y1, area->y2);

    esp_lcd_panel_draw_bitmap(panel, area->x1, area->y1, area->x2 + 1, area->y2 + 1, color_map);
    
    // 直接在这里同步通知完成
    lv_disp_flush_ready(drv);
}

static void lvgl_tick_cb(void *arg) {
    lv_tick_inc(2);
}

static void lvgl_port_task(void *arg) {
    while (1) {
        if (xSemaphoreTake(lvgl_mux, pdMS_TO_TICKS(100)) == pdTRUE) {
            lv_timer_handler();
            xSemaphoreGive(lvgl_mux);
        }
        vTaskDelay(pdMS_TO_TICKS(10));
    }
}

static esp_err_t display_lvgl_init(void) {
    lvgl_mux = xSemaphoreCreateMutex();
    lv_init();

    // 硬件复位
    gpio_reset_pin(TFT_BL);
    gpio_set_direction(TFT_BL, GPIO_MODE_OUTPUT);
    gpio_set_level(TFT_BL, 1);
    
    gpio_reset_pin(TFT_RST);
    gpio_set_direction(TFT_RST, GPIO_MODE_OUTPUT);
    gpio_set_level(TFT_RST, 0);
    vTaskDelay(pdMS_TO_TICKS(100));
    gpio_set_level(TFT_RST, 1);
    vTaskDelay(pdMS_TO_TICKS(100));

    // SPI 总线
    spi_bus_config_t buscfg = {
        .sclk_io_num = TFT_SCLK,
        .mosi_io_num = TFT_MOSI,
        .miso_io_num = -1,
        .quadwp_io_num = -1,
        .quadhd_io_num = -1,
        .max_transfer_sz = LCD_H_RES * LCD_V_RES * 2
    };
    ESP_ERROR_CHECK(spi_bus_initialize(SPI2_HOST, &buscfg, SPI_DMA_CH_AUTO));

    // 接口配置
    esp_lcd_panel_io_handle_t io_handle = NULL;
    esp_lcd_panel_io_spi_config_t io_cfg = {
        .dc_gpio_num = TFT_DC,
        .cs_gpio_num = TFT_CS,
        .pclk_hz = 10 * 1000 * 1000, 
        .spi_mode = 3,
        .lcd_cmd_bits = 8,
        .lcd_param_bits = 8,
        .trans_queue_depth = 10,
    };
    ESP_ERROR_CHECK(esp_lcd_new_panel_io_spi((esp_lcd_spi_bus_handle_t)SPI2_HOST, &io_cfg, &io_handle));

    // 驱动安装
    esp_lcd_panel_dev_config_t panel_cfg = {
        .reset_gpio_num = -1,
        .rgb_ele_order = LCD_RGB_ELEMENT_ORDER_RGB,
        .bits_per_pixel = 16,
    };
    ESP_ERROR_CHECK(esp_lcd_new_panel_st7789(io_handle, &panel_cfg, &panel_handle));
    ESP_ERROR_CHECK(esp_lcd_panel_reset(panel_handle));
    ESP_ERROR_CHECK(esp_lcd_panel_init(panel_handle));
    
    // 【修改点 A】：切换颜色反转测试
    // 如果黑屏，尝试将 true 改为 false
    ESP_ERROR_CHECK(esp_lcd_panel_invert_color(panel_handle, true)); 

    // 【修改点 B】：设置偏移量
    // 重点！！ 7 针 ST7789 240x240 绝大部分需要 Y 轴偏移 80
    // 如果 0,0 是黑屏，请务必尝试 0, 80
    ESP_ERROR_CHECK(esp_lcd_panel_set_gap(panel_handle, 0, 0)); 
    
    ESP_ERROR_CHECK(esp_lcd_panel_disp_on_off(panel_handle, true));

    // 分配缓冲区
    size_t buf_size = LCD_H_RES * LCD_DRAW_BUF_HEIGHT * sizeof(lv_color_t);
    lv_color_t *buf1 = heap_caps_malloc(buf_size, MALLOC_CAP_DMA | MALLOC_CAP_INTERNAL);
    lv_disp_draw_buf_init(&draw_buf, buf1, NULL, LCD_H_RES * LCD_DRAW_BUF_HEIGHT);

    // 注册驱动
    lv_disp_drv_init(&disp_drv);
    disp_drv.hor_res = LCD_H_RES;
    disp_drv.ver_res = LCD_V_RES;
    disp_drv.flush_cb = lvgl_flush_cb;
    disp_drv.draw_buf = &draw_buf;
    disp_drv.user_data = panel_handle;
    lv_disp_drv_register(&disp_drv);

    // Tick 定时器
    const esp_timer_create_args_t tick_args = { .callback = lvgl_tick_cb, .name = "lvgl_tick" };
    esp_timer_handle_t tick_timer;
    esp_timer_create(&tick_args, &tick_timer);
    esp_timer_start_periodic(tick_timer, 2000); 

    xTaskCreatePinnedToCore(lvgl_port_task, "lvgl_task", 4096 * 2, NULL, 5, NULL, 1);

    ESP_LOGI(TAG, "LVGL 初始化完成，当前偏移 ");
    return ESP_OK;
}

esp_err_t lcd_init_show(void){
         // 1. 初始化驱动
    display_lvgl_init();
    vTaskDelay(pdMS_TO_TICKS(1000));

    // 2. 创建 UI (必须加锁)
    if (xSemaphoreTake(lvgl_mux, portMAX_DELAY) == pdTRUE) {
        
        // 1. 获取当前活动屏幕
        lv_obj_t * screen = lv_scr_act();
        
        // --- 设置背景为白色 (White) ---
        lv_obj_set_style_bg_color(screen, lv_color_hex(0xFFFFFF), 0);
        lv_obj_set_style_bg_opa(screen, LV_OPA_COVER, 0);

        // 2. 创建标签
        lv_obj_t * label_cn = lv_label_create(screen);
        
        // --- 设置字体 (Simsun 16 CJK) ---
        lv_obj_set_style_text_font(label_cn, &lv_font_simsun_16_cjk, 0);
        
        // --- 设置中文内容 (确保 VS Code 右下角显示为 UTF-8) ---
        lv_label_set_text(label_cn, "电压：");
        
        // --- 设置文字颜色为黑色 (Black) ---
        lv_obj_set_style_text_color(label_cn, lv_color_hex(0x000000), 0);
        
        // 3. 居中显示
        lv_obj_center(label_cn);

        xSemaphoreGive(lvgl_mux);
    }
    return ESP_OK;
}