#include <stdio.h>
#include <string.h>
#include "nvs_flash.h"
#include "esp_wifi.h"
#include "esp_event.h"
#include "esp_log.h"
#include "esp_netif.h"
#include "wifidevnum_config.h"
#include "mqttconf_commun.h"
#include "betteryread.h"
#include "gpiocontrol.h"
#include "serial.h"
#include "ws2812_control.h"
#include "ota.h"
#include "lcd.h"
#include "esp_lcd_panel_ops.h"

static const char *TAG = "MAIN";
static bool init_task_triggered = false;

LV_FONT_DECLARE(lv_font_simsun_16_cjk);

static void wifi_event_handler(void* arg, esp_event_base_t base, int32_t id, void* data) {
    if (base == IP_EVENT && id == IP_EVENT_STA_GOT_IP) {
        ip_event_got_ip_t* event = (ip_event_got_ip_t*) data;
        ESP_LOGI(TAG, "成功获取 IP: " IPSTR, IP2STR(&event->ip_info.ip));

        char devid_check[32] = {0};
        // 检查 NVS 是否有设备号
        if (read_from_nvs("device_id", devid_check, sizeof(devid_check)) == ESP_OK && strlen(devid_check) > 0) {
            if (!init_task_triggered) {
                ESP_LOGI(TAG, "条件达成：网络+设备号已就绪，启动 MQTT 流程...");
                mqtt_manager_init_sequence(); 
                init_task_triggered = true;
            }
        } else {
            ESP_LOGE(TAG, "警告：无法启动 MQTT，因为 NVS 中 device_id 为空！");
        }
    } else if (base == WIFI_EVENT && id == WIFI_EVENT_STA_DISCONNECTED) {
        ESP_LOGW(TAG, "WiFi 断开，尝试重连...");
        esp_wifi_connect();
    }
}

esp_err_t device_type_init(void){
    #if defined(CONFIG_TYPE_CAR_0102)
        ESP_ERROR_CHECK(board_adc_init());
        ESP_ERROR_CHECK(gpiocontrol_init()); 
        ESP_LOGI(TAG, "使用CAR_0102");

    #elif defined(TYPE_ROBOT_0502)
        ESP_ERROR_CHECK(serial_init());//初始化串口
        ESP_LOGI(TAG, "使用仅串口控制模式");
        xTaskCreate(serial_receive_task, "serial_rx", 4096, (void*)wifi_ret, 10, NULL);
        ESP_LOGI(TAG, "使用ROBOT_0502");
    #else
        #error "请选择正确的设备型号"
    #endif

    return ESP_OK; 
} 



void app_main(void) {
    //vTaskDelay(pdMS_TO_TICKS(10000));
    
    esp_err_t ret = nvs_flash_init();

    if (ret == ESP_ERR_NVS_NO_FREE_PAGES || ret == ESP_ERR_NVS_NEW_VERSION_FOUND) {
        ESP_ERROR_CHECK(nvs_flash_erase());
        ret = nvs_flash_init();
    }
    ESP_ERROR_CHECK(ret);

    char ssid[32] = {0}, pass[64] = {0};
    esp_err_t wifi_ret=read_from_nvs("wifi_ssid", ssid, sizeof(ssid));

    ESP_ERROR_CHECK(init_spiffs()); // 启动 Web 前挂载 SPIFFS
   
    ESP_ERROR_CHECK(esp_netif_init());
    ESP_ERROR_CHECK(esp_event_loop_create_default());

    ESP_ERROR_CHECK(ws2812_init());
    
    if (wifi_ret == ESP_OK) {
        read_from_nvs("wifi_pass", pass, sizeof(pass)); 

        ESP_ERROR_CHECK(device_type_init());
        
        // 关键：创建 STA 模式的网络接口句柄
        esp_netif_create_default_wifi_sta(); 

        esp_event_handler_instance_register(IP_EVENT, IP_EVENT_STA_GOT_IP, &wifi_event_handler, NULL, NULL);
        esp_event_handler_instance_register(WIFI_EVENT, WIFI_EVENT_STA_DISCONNECTED, &wifi_event_handler, NULL, NULL);

        button_monitor_task_init();
        
        wifi_init_config_t cfg = WIFI_INIT_CONFIG_DEFAULT();
        esp_wifi_init(&cfg);
        
        wifi_config_t sta_cfg = {
            .sta = {
                .scan_method = WIFI_ALL_CHANNEL_SCAN,
                .sort_method = WIFI_CONNECT_AP_BY_SIGNAL,
            },
        };
        strncpy((char*)sta_cfg.sta.ssid, ssid, sizeof(sta_cfg.sta.ssid));
        strncpy((char*)sta_cfg.sta.password, pass, sizeof(sta_cfg.sta.password));
        
        esp_wifi_set_mode(WIFI_MODE_STA);
        esp_wifi_set_config(WIFI_IF_STA, &sta_cfg);
        esp_wifi_start();
        esp_wifi_connect();
        

        
    }else{
        ESP_LOGI(TAG, "无保存wifi 信息，启动wifi 配置页面");
        ws2812_set_color(155, 0, 0);
        nowifidata_start_config_web();
    }
} 