#include "gpiocontrol.h"
#include "esp_log.h"

static const char *TAG = "GPIO_CONTROL";

// 定义 LEDC 通道结构体
static const ledc_channel_config_t ledc_channels[] = {
    {
        .gpio_num       = GPIO_PWM_MOTOR_6,
        .speed_mode     = LEDC_MODE,
        .channel        = LEDC_CHANNEL_0,
        .intr_type      = LEDC_INTR_DISABLE,
        .timer_sel      = LEDC_TIMER,
        .duty           = 0, // 初始停止
        .hpoint         = 0
    },
    {
        .gpio_num       = GPIO_PWM_MOTOR_7,
        .speed_mode     = LEDC_MODE,
        .channel        = LEDC_CHANNEL_1,
        .intr_type      = LEDC_INTR_DISABLE,
        .timer_sel      = LEDC_TIMER,
        .duty           = 0, // 初始停止
        .hpoint         = 0
    },
    {
        .gpio_num       = GPIO_PWM_SERVO_10,
        .speed_mode     = LEDC_MODE,
        .channel        = LEDC_CHANNEL_2,
        .intr_type      = LEDC_INTR_DISABLE,
        .timer_sel      = LEDC_TIMER,
        .duty           = SERVO_INIT_90_DEG, // 舵机初始化为 90 度
        .hpoint         = 0
    },
    {
        .gpio_num       = GPIO_PWM_ESC_15,
        .speed_mode     = LEDC_MODE,
        .channel        = LEDC_CHANNEL_3,
        .intr_type      = LEDC_INTR_DISABLE,
        .timer_sel      = LEDC_TIMER,
        .duty           = ESC_MIN_DUTY, 
        .hpoint         = 0
    },
    {
        .gpio_num       = GPIO_PWM_ESC_16,
        .speed_mode     = LEDC_MODE,
        .channel        = LEDC_CHANNEL_4,
        .intr_type      = LEDC_INTR_DISABLE,
        .timer_sel      = LEDC_TIMER,
        .duty           = ESC_MIN_DUTY, 
        .hpoint         = 0
    }
};

esp_err_t gpiocontrol_init(void) {
    // 1. 配置定时器
    ledc_timer_config_t ledc_timer = {
        .speed_mode       = LEDC_MODE,
        .timer_num        = LEDC_TIMER,
        .duty_resolution  = LEDC_DUTY_RES,
        .freq_hz          = FREQUENCY,
        .clk_cfg          = LEDC_AUTO_CLK
    };
    ESP_ERROR_CHECK(ledc_timer_config(&ledc_timer));

    // 2. 配置 3 个通道
    for (int i = 0; i < 5; i++) {
        ESP_ERROR_CHECK(ledc_channel_config(&ledc_channels[i]));
    }

    ESP_LOGI(TAG, "PWM 初始化完成 (GPIO 6, 7 为电机, GPIO 10 为舵机 90度)");
    return ESP_OK;
}


void set_servo_angle(uint32_t angle) {
    if (angle > 180) angle = 180;
    
    // 将 0-180 度线性映射到 SERVO_MIN_DUTY 和 SERVO_MAX_DUTY 之间
    uint32_t duty = SERVO_MIN_DUTY + (angle * (SERVO_MAX_DUTY - SERVO_MIN_DUTY) / 180);
    
    ledc_set_duty(LEDC_MODE, LEDC_CHANNEL_2, duty);
    ledc_update_duty(LEDC_MODE, LEDC_CHANNEL_2);
}

/**
 * @brief 设置电机驱动引脚占空比 (GPIO 6, 7)
 * @param gpio GPIO_PWM_MOTOR_6 或 GPIO_PWM_MOTOR_7
 * @param duty_percent 0 到 100 之间的百分比
 */
void set_motor_speed(int gpio, uint32_t duty_percent) {
    if (duty_percent > 100) duty_percent = 100;
    
    // 百分比转换为 13-bit 数值 (0-8191)
    uint32_t duty = (duty_percent * 8191) / 100;
    
    ledc_channel_t channel = (gpio == GPIO_PWM_MOTOR_6) ? LEDC_CHANNEL_0 : LEDC_CHANNEL_1;
    
    ledc_set_duty(LEDC_MODE, channel, duty);
    ledc_update_duty(LEDC_MODE, channel);
}

void car0102_speed_stop(void) {
    // 将电机引脚占空比降为 0
    ledc_set_duty(LEDC_MODE, LEDC_CHANNEL_0, 0);
    ledc_update_duty(LEDC_MODE, LEDC_CHANNEL_0);
    ledc_set_duty(LEDC_MODE, LEDC_CHANNEL_1, 0);
    ledc_update_duty(LEDC_MODE, LEDC_CHANNEL_1);
    ESP_LOGD(TAG, "电机已强制停止输出");
}


void car0102_control_motor(int mode ,int val){

    if(mode == 1){
        if(val <50){
            set_motor_speed(GPIO_PWM_MOTOR_6,0);
            set_motor_speed(GPIO_PWM_MOTOR_7,0);
        }else{
            set_motor_speed(GPIO_PWM_MOTOR_6,val/2-10);
            set_motor_speed(GPIO_PWM_MOTOR_7,0);
        }

        
    }else if(mode == 2){
        if(val <50){
            set_motor_speed(GPIO_PWM_MOTOR_6,0);
            set_motor_speed(GPIO_PWM_MOTOR_7,0);
        }else{
            set_motor_speed(GPIO_PWM_MOTOR_6,0);
            set_motor_speed(GPIO_PWM_MOTOR_7,val/2-10);
        }
    }
    if(mode == 3){
        if(val <45) set_servo_angle(90);
        else if(val<70) set_servo_angle(50+val+7);
        else if(val>=70) set_servo_angle(135);
    }else if(mode == 4){
         if(val <45) set_servo_angle(90);
        else if(val<70) set_servo_angle(130-val-7);
        else if(val>=70) set_servo_angle(45);
    }
}

void esc_set_throttle(uint8_t percent)
{
    if (percent > 100) percent = 100;

    uint32_t duty = ESC_MIN_DUTY +
        (ESC_MAX_DUTY - ESC_MIN_DUTY) * percent / 100;

    ledc_set_duty(LEDC_LOW_SPEED_MODE, LEDC_CHANNEL_3, duty);
    ledc_update_duty(LEDC_LOW_SPEED_MODE, LEDC_CHANNEL_3);

    ledc_set_duty(LEDC_LOW_SPEED_MODE, LEDC_CHANNEL_4, duty);
    ledc_update_duty(LEDC_LOW_SPEED_MODE, LEDC_CHANNEL_4);
}