#include "wifidevnum_config.h"
#include <stdio.h>
#include <string.h>
#include <ctype.h> 
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "esp_wifi.h"
#include "esp_event.h"
#include "esp_log.h"
#include "esp_http_server.h"
#include "driver/gpio.h"
#include "lwip/sockets.h"
#include "nvs_flash.h"
#include "nvs.h"
#include "esp_spiffs.h"
#include "mqttconf_commun.h"
#include "serial.h"

static const char *TAG = "WIFI_CFG";
static bool wifi_editable = false;
static bool devid_editable = false;

// 初始化 SPIFFS
esp_err_t init_spiffs(void) {
    esp_vfs_spiffs_conf_t conf = {
        .base_path = "/spiffs",
        .partition_label = "storage",
        .max_files = 5,
        .format_if_mount_failed = true
    };
    esp_err_t ret = esp_vfs_spiffs_register(&conf);
    if (ret != ESP_OK) {
        ESP_LOGE(TAG, "SPIFFS 挂载失败: %s", esp_err_to_name(ret));
        return ret;
    }
    return ESP_OK;
}

// URL 解码
void url_decode(char *dst, const char *src) {
    char a, b;
    while (*src) {
        if ((*src == '%') && ((a = src[1]) && (b = src[2])) && (isxdigit(a) && isxdigit(b))) {
            a = toupper((unsigned char)a); b = toupper((unsigned char)b);
            a -= (a >= 'A') ? ('A' - 10) : '0';
            b -= (b >= 'A') ? ('A' - 10) : '0';
            *dst++ = (char)((a << 4) | b);
            src += 3;
        } else if (*src == '+') {
            *dst++ = ' '; src++;
        } else {
            *dst++ = *src++;
        }
    }
    *dst = '\0';
}

void save_to_nvs(const char* key, const char* value) {
    nvs_handle_t handle;
    if (nvs_open("storage", NVS_READWRITE, &handle) == ESP_OK) {
        nvs_set_str(handle, key, value);
        nvs_commit(handle);
        nvs_close(handle);
        ESP_LOGI(TAG, "NVS 写入成功 [%s]", key);
    }
}

esp_err_t read_from_nvs(const char* key, char* buf, size_t len) {
    nvs_handle_t handle;
    esp_err_t err = nvs_open("storage", NVS_READONLY, &handle);
    if (err == ESP_OK) {
        err = nvs_get_str(handle, key, buf, &len);
        nvs_close(handle);
    }
    return err;
}

static esp_err_t get_handler(httpd_req_t *req) {
    char wifi_status[128], devid_status[128];
    sprintf(wifi_status, wifi_editable ? "" : "disabled style='background:#eee'");
    sprintf(devid_status, devid_editable ? "" : "disabled style='background:#eee'");

    // 1. 从 SPIFFS 读取 HTML 模板
    FILE* f = fopen("/spiffs/index.html", "r");
    if (!f) {
        httpd_resp_send_404(req);
        return ESP_FAIL;
    }
    fseek(f, 0, SEEK_END);
    long fsize = ftell(f);
    fseek(f, 0, SEEK_SET);

    char *template = malloc(fsize + 1);
    fread(template, 1, fsize, f);
    fclose(f);
    template[fsize] = 0;

    // 2. 准备最终网页缓冲区 (S3 N16R8 内存大，直接给足够空间)
    char *final_html = malloc(fsize + 1024);
    snprintf(final_html, fsize + 1024, template,
             wifi_editable ? "🔓":"🔒", wifi_status, wifi_status, 
             devid_editable ? "🔓":"🔒", devid_status);

    httpd_resp_set_type(req, "text/html");
    httpd_resp_send(req, final_html, HTTPD_RESP_USE_STRLEN);

    free(template);
    free(final_html);
    return ESP_OK;
}

static esp_err_t post_handler(httpd_req_t *req) {
    char buf[512];
    int ret = httpd_req_recv(req, buf, req->content_len);
    if (ret > 0) {
        buf[ret] = '\0';
        char *token, *saveptr;
        token = strtok_r(buf, "&", &saveptr);
        while (token != NULL) {
            char *key = token;
            char *val = strchr(token, '=');
            if (val) {
                *val = '\0'; val++;
                char decoded_val[64];
                url_decode(decoded_val, val);
                if (strcmp(key, "ssid") == 0 && wifi_editable) save_to_nvs("wifi_ssid", decoded_val);
                else if (strcmp(key, "pass") == 0 && wifi_editable) save_to_nvs("wifi_pass", decoded_val);
                else if (strcmp(key, "devid") == 0 && devid_editable) save_to_nvs("device_id", decoded_val);
            }
            token = strtok_r(NULL, "&", &saveptr);
        }
    }
    httpd_resp_send(req, "<h1>Success</h1><p>Restarting...</p>", HTTPD_RESP_USE_STRLEN);
    vTaskDelay(pdMS_TO_TICKS(1500));
    esp_restart();
    return ESP_OK;
}

void dns_server_task(void *pvParameters) {
    uint8_t rx_buffer[128];
    struct sockaddr_in dest_addr;
    int sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_IP);
    dest_addr.sin_addr.s_addr = htonl(INADDR_ANY);
    dest_addr.sin_family = AF_INET;
    dest_addr.sin_port = htons(53);
    bind(sock, (struct sockaddr *)&dest_addr, sizeof(dest_addr));
    while (1) {
        struct sockaddr_in source_addr;
        socklen_t socklen = sizeof(source_addr);
        int len = recvfrom(sock, rx_buffer, sizeof(rx_buffer), 0, (struct sockaddr *)&source_addr, &socklen);
        if (len > 12) {
            rx_buffer[2] |= 0x80; rx_buffer[3] |= 0x80; rx_buffer[7] = 1;
            uint8_t answer[] = { 0xc0, 0x0c, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x04, 192, 168, 4, 1 };
            memcpy(rx_buffer + len, answer, sizeof(answer));
            sendto(sock, rx_buffer, len + sizeof(answer), 0, (struct sockaddr *)&source_addr, sizeof(source_addr));
        }
    }
}

esp_err_t http_404_error_handler(httpd_req_t *req, httpd_err_code_t err) {
    httpd_resp_set_status(req, "302 Found");
    httpd_resp_set_hdr(req, "Location", "/");
    httpd_resp_send(req, NULL, 0);
    return ESP_OK;
}

void start_config_web() {
    esp_wifi_stop();
    esp_netif_create_default_wifi_ap();
    wifi_init_config_t cfg = WIFI_INIT_CONFIG_DEFAULT();
    esp_wifi_init(&cfg);
    wifi_config_t ap_cfg = {
        .ap = { .ssid = CONFIG_ROBIOT_WIFI_SSID, .max_connection = 4, .authmode = WIFI_AUTH_OPEN }
    };
    esp_wifi_set_mode(WIFI_MODE_AP);
    esp_wifi_set_config(WIFI_IF_AP, &ap_cfg);
    esp_wifi_start();
    xTaskCreate(dns_server_task, "dns", 3072, NULL, 5, NULL);

    httpd_handle_t server = NULL;
    httpd_config_t config = HTTPD_DEFAULT_CONFIG();
    if (httpd_start(&server, &config) == ESP_OK) {
        httpd_uri_t get_uri = { .uri="/", .method=HTTP_GET, .handler=get_handler };
        httpd_register_uri_handler(server, &get_uri);
        httpd_uri_t post_uri = { .uri="/save", .method=HTTP_POST, .handler=post_handler };
        httpd_register_uri_handler(server, &post_uri);
        httpd_register_err_handler(server, HTTPD_404_NOT_FOUND, http_404_error_handler);
    }
}

void nowifidata_start_config_web(void) { 
    wifi_editable = true;
    devid_editable = true;
    start_config_web();
}

void button_monitor_task(void* arg) {

    gpio_config_t io_conf = {
        .pin_bit_mask = (1ULL << BTN_WIFI_IO) | (1ULL << BTN_DEVID_IO),
        .mode = GPIO_MODE_INPUT, .pull_up_en = 1
    };
    gpio_config(&io_conf);
    int wifi_cnt = 0, dev_cnt = 0;
    bool web_started = false;
    while (1) {
        if (gpio_get_level(BTN_WIFI_IO) == 0) {
            if (++wifi_cnt == 30) { 
                wifi_editable = true; 
                if(!web_started){ 
                    mqtt_manager_stop();    // 停止 MQTT
                    set_serial_enabled(false); // 停止串口主动发送
                    start_config_web(); 
                    web_started=true; 
                }
            }
        } else wifi_cnt = 0;
        
        if (gpio_get_level(BTN_DEVID_IO) == 0) {
            if (++dev_cnt == 30) { 
                devid_editable = true; 
                if(!web_started){ 
                     mqtt_manager_stop();
                     set_serial_enabled(false); // 停止串口主动发送
                    start_config_web(); 
                    web_started=true; 
                }
            }
        } else dev_cnt = 0;
        vTaskDelay(pdMS_TO_TICKS(100));
    }
}