#include "betteryread.h"
#include "esp_adc/adc_oneshot.h"
#include "esp_adc/adc_cali.h"
#include "esp_adc/adc_cali_scheme.h"
#include "esp_log.h"

static const char *TAG = "ADC_CONTROL";

// 全局句柄
static adc_oneshot_unit_handle_t adc1_handle;
static adc_cali_handle_t adc_cali_handle = NULL;
static bool do_calibration = false;


// 校准初始化函数（内部使用）
static bool example_adc_calibration_init(adc_unit_t unit, adc_channel_t channel, adc_atten_t atten, adc_cali_handle_t *out_handle) {
    adc_cali_handle_t handle = NULL;
    esp_err_t ret = ESP_FAIL;
    bool calibrated = false;

    // ESP32-S3 支持 Curve Fitting 校准方式
    if (!calibrated) {
        adc_cali_curve_fitting_config_t cali_config = {
            .unit_id = unit,
            .chan = channel,
            .atten = atten,
            .bitwidth = ADC_BITWIDTH_DEFAULT,
        };
        ret = adc_cali_create_scheme_curve_fitting(&cali_config, &handle);
        if (ret == ESP_OK) {
            calibrated = true;
        }
    }
    *out_handle = handle;
    ESP_LOGI(TAG, "ADC初始化成功");
    return calibrated;
}

esp_err_t board_adc_init(void) {
    adc_oneshot_unit_init_cfg_t init_config1 = {
        .unit_id = ADC_UNIT_1,
    };
    ESP_ERROR_CHECK(adc_oneshot_new_unit(&init_config1, &adc1_handle));

    adc_oneshot_chan_cfg_t config = {
        .bitwidth = ADC_BITWIDTH_DEFAULT,
        .atten = ADC_ATTEN_DB_12, 
    };
    ESP_ERROR_CHECK(adc_oneshot_config_channel(adc1_handle, ADC_CHANNEL_4, &config));

    // 初始化校准
    do_calibration = example_adc_calibration_init(ADC_UNIT_1, ADC_CHANNEL_4, ADC_ATTEN_DB_12, &adc_cali_handle);
    return ESP_OK;
}

/**
 * @brief 读取当前电压值
 * @return int 返回毫伏值 (mV)，读取失败返回 -1
 */
// 核心修改：返回 float
float board_get_voltage_v(void) {
    int adc_raw;
    int voltage_mv;

    esp_err_t ret = adc_oneshot_read(adc1_handle, ADC_CHANNEL_4, &adc_raw);
    if (ret != ESP_OK) return -1.0f;

    if (do_calibration) {
        // 先获取校准后的毫伏值 (int)
        adc_cali_raw_to_voltage(adc_cali_handle, adc_raw, &voltage_mv);
        // 转换为伏特并返回小数
        return (float)voltage_mv / 1000.0f;
    } else {
        return (float)(adc_raw * 3100) / 4095.0f / 1000.0f;
    }
}