#include "serialota.h"
#include "serial.h"  // 使用你提供的串口定义
#include "esp_http_client.h"
#include "esp_partition.h"
#include "esp_log.h"
#include "driver/uart.h"
#include <string.h>

static const char *TAG = "STM32_OTA";

// 分区元数据
#define PARTITION_SUBTYPE_STM32  0x40
#define PARTITION_NAME           "stm32ota"

/**
 * HTTP 事件处理：将下载的数据流写入 Flash 分区
 */
static esp_err_t _http_event_handler(esp_http_client_event_t *evt) {
    static const esp_partition_t *partition = NULL;
    static size_t written_len = 0;

    switch (evt->event_id) {
        case HTTP_EVENT_ON_CONNECTED:
            partition = esp_partition_find_first(ESP_PARTITION_TYPE_DATA, PARTITION_SUBTYPE_STM32, PARTITION_NAME);
            if (!partition) {
                ESP_LOGE(TAG, "找不到 stm32ota 分区");
                return ESP_FAIL;
            }
            // 擦除分区（简单处理：全擦除。实际可根据 content-length 擦除）
            esp_partition_erase_range(partition, 0, partition->size);
            written_len = 0;
            break;

        case HTTP_EVENT_ON_DATA:
            if (partition && evt->data_len > 0) {
                esp_partition_write(partition, written_len, evt->data, evt->data_len);
                written_len += evt->data_len;
                ESP_LOGD(TAG, "已下载: %d bytes", written_len);
            }
            break;

        default:
            break;
    }
    return ESP_OK;
}

esp_err_t stm32_ota_download_from_url(const char *url) {
    esp_http_client_config_t config = {
        .url = url,
        .event_handler = _http_event_handler,
        .timeout_ms = 10000,
    };
    esp_http_client_handle_t client = esp_http_client_init(&config);
    esp_err_t err = esp_http_client_perform(client);
    
    if (err == ESP_OK) {
        ESP_LOGI(TAG, "HEX 文件下载完成");
    } else {
        ESP_LOGE(TAG, "下载失败: %s", esp_err_to_name(err));
    }
    esp_http_client_cleanup(client);
    return err;
}

/**
 * 简单的 Intel HEX 字符转字节
 */
static uint8_t hex2byte(const char *hex) {
    uint8_t byte = 0;
    for (int i = 0; i < 2; i++) {
        char c = hex[i];
        byte <<= 4;
        if (c >= '0' && c <= '9') byte |= (c - '0');
        else if (c >= 'A' && c <= 'F') byte |= (c - 'A' + 10);
        else if (c >= 'a' && c <= 'f') byte |= (c - 'a' + 10);
    }
    return byte;
}

/**
 * 核心烧录逻辑：读取分区 -> 解析 HEX -> 串口发送
 */
esp_err_t stm32_ota_start_update(void) {
    const esp_partition_t *partition = esp_partition_find_first(ESP_PARTITION_TYPE_DATA, PARTITION_SUBTYPE_STM32, PARTITION_NAME);
    if (!partition) return ESP_FAIL;

    // 1. 禁用普通的串口接收任务，防止冲突
    set_serial_enabled(false);
    vTaskDelay(pdMS_TO_TICKS(100));

    // 2. 发送开始升级指令 (自定义协议)
    // 假设 mode=0xF0, value=0x01 是进入 Bootloader 的指令
    serial_send_command(0xF0, 0x01, 0, 0, 0);
    vTaskDelay(pdMS_TO_TICKS(500)); // 等待 STM32 擦除 Flash

    char line[128];
    size_t offset = 0;
    uint8_t bin_buf[32];

    ESP_LOGI(TAG, "开始解析并发送数据...");

    // 循环读取分区里的 HEX 文件行
    // 注意：HEX 文件是以 ':' 开头的文本行
    while (offset < partition->size) {
        // 简化处理：逐字节查找换行符读取一行
        int line_ptr = 0;
        char c;
        while (line_ptr < 127) {
            esp_partition_read(partition, offset++, &c, 1);
            if (c == '\n' || c == '\r') {
                if (line_ptr > 0) break; 
                else continue;
            }
            line[line_ptr++] = c;
            if (offset >= partition->size) break;
        }
        line[line_ptr] = '\0';

        if (line[0] == ':') {
            // 解析 Intel HEX
            uint8_t len = hex2byte(&line[1]);
            uint8_t type = hex2byte(&line[7]);
            
            if (type == 0x00) { // 数据记录
                for (int i = 0; i < len; i++) {
                    bin_buf[i] = hex2byte(&line[9 + i * 2]);
                }
                
                // 通过串口发送二进制块给 STM32
                uart_write_bytes(STM32_UART_PORT, (const char *)bin_buf, len);
                
                // 等待 STM32 的应答 (ACK: 0x06)
                uint8_t ack = 0;
                int rx_len = uart_read_bytes(STM32_UART_PORT, &ack, 1, pdMS_TO_TICKS(500));
                if (rx_len <= 0 || ack != 0x06) {
                    ESP_LOGE(TAG, "STM32 无响应或校验失败");
                    goto error;
                }
            } else if (type == 0x01) { // 文件结束记录
                ESP_LOGI(TAG, "解析完成");
                break;
            }
        }
    }

    // 3. 结束升级，跳转 APP
    serial_send_command(0xF0, 0x02, 0, 0, 0); 
    set_serial_enabled(true);
    return ESP_OK;

error:
    set_serial_enabled(true);
    return ESP_FAIL;
}