#include "ws2812_control.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "esp_log.h"
#include "esp_system.h"
#include "esp_heap_caps.h"      // 必须包含：用于 heap_caps_calloc
#include "freertos/FreeRTOS.h"  // 必须包含：用于 portMAX_DELAY
#include "driver/rmt_tx.h"
#include "driver/gpio.h"

static const char *TAG = "ws2812";

// 全局变量
static rmt_channel_handle_t tx_chan = NULL;
static rmt_encoder_handle_t encoder = NULL;
static uint8_t *led_buffer = NULL;
static int led_count = 0;

// 字节编码器配置
static rmt_bytes_encoder_config_t bytes_encoder_config = {
    .bit0 = {
        .level0 = 1, .duration0 = T0H_NS * RMT_TICK_PER_US / 1000,
        .level1 = 0, .duration1 = T0L_NS * RMT_TICK_PER_US / 1000,
    },
    .bit1 = {
        .level0 = 1, .duration0 = T1H_NS * RMT_TICK_PER_US / 1000,
        .level1 = 0, .duration1 = T1L_NS * RMT_TICK_PER_US / 1000,
    },
    .flags.msb_first = 1
};

static void ws2812_deinit(void) {
    if (encoder) rmt_del_encoder(encoder);
    if (tx_chan) {
        rmt_disable(tx_chan);
        rmt_del_channel(tx_chan);
    }
    if (led_buffer) free(led_buffer);
    encoder = NULL;
    tx_chan = NULL;
    led_buffer = NULL;
    led_count = 0;
}

esp_err_t ws2812_init(void) {
    esp_err_t ret;
    ws2812_deinit();
    led_count = LED_COUNT;
    
    // 修正：使用 LED_COUNT 宏，并包含 esp_heap_caps.h
    led_buffer = heap_caps_calloc(led_count, 3, MALLOC_CAP_DMA);
    if (!led_buffer) {
        ESP_LOGE(TAG, "内存分配失败");
        return ESP_ERR_NO_MEM;
    }
    
    rmt_tx_channel_config_t tx_chan_config = {
        .clk_src = RMT_CLK_SRC_DEFAULT,
        .gpio_num = RGB_PIN,           // 修正：对应 .h 中的宏
        .mem_block_symbols = 64,
        .resolution_hz = RMT_RESOLUTION_HZ,
        .trans_queue_depth = 4,
        .flags.invert_out = false,
    };
    
    ret = rmt_new_tx_channel(&tx_chan_config, &tx_chan);
    if (ret != ESP_OK) {
        ESP_LOGE(TAG, "创建RMT通道失败");
        free(led_buffer);
        return ret;
    }
    
    ret = rmt_new_bytes_encoder(&bytes_encoder_config, &encoder);
    if (ret != ESP_OK) {
        ESP_LOGE(TAG, "创建编码器失败");
        rmt_del_channel(tx_chan);
        free(led_buffer);
        return ret;
    }
    
    ret = rmt_enable(tx_chan);
    if (ret != ESP_OK) {
        rmt_del_encoder(encoder);
        rmt_del_channel(tx_chan);
        free(led_buffer);
        return ret;
    }
    
    // 修正：打印变量名统一
    ESP_LOGI(TAG, "WS2812初始化完成, GPIO: %d, LED数量: %d", RGB_PIN, LED_COUNT);
    return ESP_OK;
}

void ws2812_set_color(uint8_t r, uint8_t g, uint8_t b) {
    if (!led_buffer || !tx_chan || !encoder) return;
    
    for (int i = 0; i < led_count; i++) {
        int idx = i * 3;
        led_buffer[idx] = g;      
        led_buffer[idx + 1] = r;  
        led_buffer[idx + 2] = b;  
    }
    
    rmt_transmit_config_t tx_config = {
        .loop_count = 0,
        .flags.eot_level = 0,
    };
    
    // 现在 portMAX_DELAY 可以识别了
    rmt_transmit(tx_chan, encoder, led_buffer, led_count * 3, &tx_config);
    rmt_tx_wait_all_done(tx_chan, portMAX_DELAY);
}

void ws2812_clear(void) {
    ws2812_set_color(0, 0, 0);
}

