#include <stdio.h>
#include <string.h>
#include "nvs_flash.h"
#include "esp_wifi.h"
#include "esp_event.h"
#include "esp_log.h"
#include "esp_netif.h"
#include "wifidevnum_config.h"
#include "mqttconf_commun.h"
#include "serial.h"

static const char *TAG = "MAIN";
static bool init_task_triggered = false;

static void wifi_event_handler(void* arg, esp_event_base_t base, int32_t id, void* data) {
    if (base == IP_EVENT && id == IP_EVENT_STA_GOT_IP) {
        ip_event_got_ip_t* event = (ip_event_got_ip_t*) data;
        ESP_LOGI(TAG, "成功获取 IP: " IPSTR, IP2STR(&event->ip_info.ip));

        char devid_check[32] = {0};
        // 检查 NVS 是否有设备号
        if (read_from_nvs("device_id", devid_check, sizeof(devid_check)) == ESP_OK && strlen(devid_check) > 0) {
            if (!init_task_triggered) {
                ESP_LOGI(TAG, "条件达成：网络+设备号已就绪，启动 MQTT 流程...");
                mqtt_manager_init_sequence(); 
                init_task_triggered = true;
            }
        } else {
            ESP_LOGE(TAG, "警告：无法启动 MQTT，因为 NVS 中 device_id 为空！");
        }
    } else if (base == WIFI_EVENT && id == WIFI_EVENT_STA_DISCONNECTED) {
        ESP_LOGW(TAG, "WiFi 断开，尝试重连...");
        esp_wifi_connect();
    }
}

void app_main(void) {
    esp_err_t ret = nvs_flash_init();
    if (ret == ESP_ERR_NVS_NO_FREE_PAGES || ret == ESP_ERR_NVS_NEW_VERSION_FOUND) {
        ESP_ERROR_CHECK(nvs_flash_erase());
        ret = nvs_flash_init();
    }
    ESP_ERROR_CHECK(ret);

    ESP_ERROR_CHECK(init_spiffs()); // 启动 Web 前挂载 SPIFFS
   
    ESP_ERROR_CHECK(esp_netif_init());
    ESP_ERROR_CHECK(esp_event_loop_create_default());
    
    ESP_ERROR_CHECK(serial_init());//初始化串口

    // 关键：创建 STA 模式的网络接口句柄
    esp_netif_create_default_wifi_sta(); 

    esp_event_handler_instance_register(IP_EVENT, IP_EVENT_STA_GOT_IP, &wifi_event_handler, NULL, NULL);
    esp_event_handler_instance_register(WIFI_EVENT, WIFI_EVENT_STA_DISCONNECTED, &wifi_event_handler, NULL, NULL);

    xTaskCreate(button_monitor_task, "btn_task", 4096, NULL, 5, NULL);

    char ssid[32] = {0}, pass[64] = {0};
    if (read_from_nvs("wifi_ssid", ssid, sizeof(ssid)) == ESP_OK) {
        read_from_nvs("wifi_pass", pass, sizeof(pass));
        
        wifi_init_config_t cfg = WIFI_INIT_CONFIG_DEFAULT();
        esp_wifi_init(&cfg);
        
        wifi_config_t sta_cfg = {
            .sta = {
                .scan_method = WIFI_ALL_CHANNEL_SCAN,
                .sort_method = WIFI_CONNECT_AP_BY_SIGNAL,
            },
        };
        strncpy((char*)sta_cfg.sta.ssid, ssid, sizeof(sta_cfg.sta.ssid));
        strncpy((char*)sta_cfg.sta.password, pass, sizeof(sta_cfg.sta.password));
        
        esp_wifi_set_mode(WIFI_MODE_STA);
        esp_wifi_set_config(WIFI_IF_STA, &sta_cfg);
        esp_wifi_start();
        esp_wifi_connect();

        xTaskCreate(serial_receive_task, "serial_rx", 4096, NULL, 10, NULL);
    }
} 