#include "serial.h"
#include <string.h>
#include "driver/uart.h"
#include "driver/gpio.h"
#include "esp_log.h"
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "esp_timer.h"

static const char *TAG = "STM32_SERIAL";

beatstatus_e beat_status =  BEAT_UNKNOWN;

// 记录最后一次收到数据的时间 (单位: 微秒)
static int64_t g_last_rx_time = 0;

static float stm32_Battery_voltage = 0;

static float stm32_tempvalue=0;

static bool g_serial_enabled = true; // <--- 新增控制位

void set_serial_enabled(bool enabled) {
    g_serial_enabled = enabled;
}

bool get_serial_enabled() {
    return g_serial_enabled;
}

//static bool g_serial_stm32_flag = false;

/**
 * @brief 串口初始化
 */
esp_err_t serial_init(void) {
    uart_config_t uart_config = {
        .baud_rate = STM32_BAUD_RATE,
        .data_bits = UART_DATA_8_BITS,
        .parity    = UART_PARITY_DISABLE,
        .stop_bits = UART_STOP_BITS_1,
        .flow_ctrl = UART_HW_FLOWCTRL_DISABLE,
        .source_clk = UART_SCLK_DEFAULT,
    };

    // 安装驱动
    ESP_ERROR_CHECK(uart_driver_install(STM32_UART_PORT, 1024 * 2, 0, 0, NULL, 0));
    // 配置参数
    ESP_ERROR_CHECK(uart_param_config(STM32_UART_PORT, &uart_config));
    // 设置引脚
    ESP_ERROR_CHECK(uart_set_pin(STM32_UART_PORT, STM32_TXD_PIN, STM32_RXD_PIN, UART_PIN_NO_CHANGE, UART_PIN_NO_CHANGE));
    
    // 初始化时间戳为当前时间，防止启动即报错
    g_last_rx_time = esp_timer_get_time();
    return ESP_OK; 
}

/**
 * @brief 发送 8 字节指令 (协议完全匹配 STM32)
 */
void serial_send_command(uint8_t mode, uint8_t value,uint8_t data3,uint8_t data4,uint8_t data5) {
    if (!g_serial_enabled) return; 
    
    uint8_t frame[8];
    frame[0] = 0x01;          // 帧头
    frame[1] = mode;          // 数据位 1
    frame[2] = value;         // 数据位 2
    frame[3] = data3;          // 预留
    frame[4] = data4;          // 预留
    frame[5] = data5;          // 预留
    
    // 生成和校验 (Byte1 + Byte2 + Byte3 + Byte4 + Byte5)
    frame[6] = (uint8_t)((frame[1] + frame[2] + frame[3] + frame[4] + frame[5]) & 0xFF);
    
    frame[7] = 0xFF;          // 停止位 (STM32 代码中 frame[7] 是 0xFF)

    uart_write_bytes(STM32_UART_PORT, (const char *)frame, 8);
}

float get_stm32_battery_voltage(){
    return stm32_Battery_voltage;
}

float get_stm32_battery_tempvalue(){
    return stm32_tempvalue;
}

void recserial_data(uint8_t *data){
    if(data[1]==0x00){
        uint16_t voltage_int = (data[4] << 8) | data[5];
        stm32_Battery_voltage = voltage_int / 100.0f;
        uint16_t temp_int = (data[2] << 8) | data[3];
        stm32_tempvalue = temp_int / 100.0f;
        ESP_LOGI(TAG, "电压: %.2fV, 温度: %.2f", stm32_Battery_voltage, stm32_tempvalue);
    }
}

/**
 * @brief 接收与超时检测任务
 */
void serial_receive_task(void *pvParameters) {
    uint8_t data[128];
    uint8_t frame_buf[8];

    serial_send_command(0x00,0x00,0x00,0x00,0x00);

    while (1) {
        if(g_serial_enabled != true){
            vTaskDelay(pdMS_TO_TICKS(100));
            break;
        }
        // 1. 读取串口数据 (非阻塞或短超时)
        int len = uart_read_bytes(STM32_UART_PORT, data, sizeof(data), pdMS_TO_TICKS(100));

        if (len > 0) {
            // 简单协议解析逻辑：寻找 0x01 帧头且长度满足 8 字节
            // 提示：在复杂环境下建议使用状态机解析
            for (int i = 0; i < len; i++) {
                if (data[i] == 0x01 && (i + 7) < len) {
                    memcpy(frame_buf, &data[i], 8);
                    
                    // 校验和验证
                    uint8_t check = (uint8_t)((frame_buf[1] + frame_buf[2] + frame_buf[3] + frame_buf[4] + frame_buf[5]) & 0xFF);
                    
                    if (frame_buf[6] == check && frame_buf[7] == 0xFF) {
                        recserial_data(frame_buf);
                        // 更新最后接收时间戳
                        g_last_rx_time = esp_timer_get_time();
                        i += 7; // 跳过处理完的帧
                    }
                }
            }
            
        }

        // 2. 超时检测 (10秒 = 10,000,000 微秒)
        int64_t now = esp_timer_get_time();
        if ((now - g_last_rx_time) > 10000000) {
            if (g_serial_enabled) { // <--- 只有启用时才发
                ESP_LOGE(TAG, "STM32 失联！尝试发送激活指令...");
                serial_send_command(0x00,0x00,0x00,0x00,0x00);
            }
            beat_status = BEAT_STOP;
            // 可选：这里可以尝试重新初始化串口或发送心跳请求
            // 为了防止日志刷屏，可以稍微加长一点检测间隔
            vTaskDelay(pdMS_TO_TICKS(3000));
        }else  beat_status = BEAT_START;

        vTaskDelay(pdMS_TO_TICKS(10)); // 防止任务空转占满 CPU
    }
}