#include "device_linked_list.hpp"// devicelist.cc

#include "mylog.hpp"
#include <sstream> // 用于在函数中高效地构建字符串

// --- DeviceNode 构造函数实现 ---
DeviceNode::DeviceNode(const std::string& id, int cool, int usage) 
    : deviceId(id), coolDownTime(cool), usageTime(usage), next(nullptr) {}

// --- DeviceList 成员函数实现 ---

// 构造函数：初始化头尾指针为空
DeviceList::DeviceList() : head(nullptr), tail(nullptr) {}

// 析构函数：遍历并释放所有节点，防止内存泄漏
DeviceList::~DeviceList() {
    DeviceNode* current = head;
    while (current != nullptr) {
        DeviceNode* nextNode = current->next;
        delete current;
        current = nextNode;
    }
    head = nullptr;
    tail = nullptr;
}

// 添加新设备
void DeviceList::addDevice(const std::string& id, int coolDown, int usage) {
    // 步骤1: 检查ID是否已存在，防止重复添加
    DeviceNode* temp = head;
    while (temp != nullptr) {
        if (temp->deviceId == id) {
            std::string msg = "错误: 设备 " + id + " 已存在，无法重复添加。";
            mylog_warn(msg.c_str());
            return;
        }
        temp = temp->next;
    }

    // 步骤2: 创建新节点
    DeviceNode* newNode = new DeviceNode(id, coolDown, usage);
    
    // 步骤3: 将新节点添加到链表尾部
    if (head == nullptr) { // 如果链表为空
        head = newNode;
        tail = newNode;
    } else { // 如果链表不为空
        tail->next = newNode;
        tail = newNode; // 更新尾指针
    }
    
    std::string msg = "设备 " + id + " 添加成功！";
    mylog_info(msg.c_str());
}

// 删除设备
bool DeviceList::deleteDevice(const std::string& id) {
    if (head == nullptr) {
        mylog_warn("链表为空，无法删除！");
        return false;
    }

    DeviceNode* current = head;
    DeviceNode* prev = nullptr;

    // 查找要删除的节点
    while (current != nullptr && current->deviceId != id) {
        prev = current;
        current = current->next;
    }

    // 如果未找到节点
    if (current == nullptr) {
        std::string msg = "未找到设备 " + id;
        mylog_warn(msg.c_str());
        return false;
    }
    
    // 处理删除逻辑
    if (prev == nullptr) { // 如果删除的是头节点
        head = current->next;
        if (head == nullptr) { // 如果删除后链表为空
            tail = nullptr;
        }
    } else { // 如果删除的是中间或尾节点
        prev->next = current->next;
        if (current == tail) { // 如果删除的是尾节点，需更新tail指针
            tail = prev;
        }
    }

    delete current; // 释放节点内存
    std::string msg = "设备 " + id + " 删除成功！";
    mylog_info(msg.c_str());
    return true;
}

// 修改设备信息
bool DeviceList::modifyDevice(const std::string& id, int newCoolDown, int newUsage) {
    DeviceNode* current = head;
    while (current != nullptr) {
        if (current->deviceId == id) {
            current->coolDownTime = newCoolDown;
            current->usageTime = newUsage;
            std::string msg = "设备 " + id + " 修改成功！";
            mylog_info(msg.c_str());
            return true;
        }
        current = current->next;
    }
    
    std::string msg = "未找到设备 " + id + "，修改失败。";
    mylog_warn(msg.c_str());
    return false;
}

// 显示所有设备信息
void DeviceList::displayAllDevices() const {
    if (head == nullptr) {
        mylog_info("链表中没有设备信息！");
        return;
    }

    mylog_info("\n========== 所有设备信息 ==========");
    mylog_info("设备号\t\t\t冷却时间(秒)\t使用时间(秒)");
    mylog_info("-----------------------------------------------------------");

    DeviceNode* current = head;
    while (current != nullptr) {
        std::stringstream ss;
        ss << current->deviceId << "\t" << current->coolDownTime << "\t\t" << current->usageTime;
        mylog_info(ss.str().c_str());
        current = current->next;
    }
    mylog_info("===========================================================");
}

// 查找指定设备
bool DeviceList::findDevice(const std::string& id) const {
    DeviceNode* current = head;
    while (current != nullptr) {
        if (current->deviceId == id) {
            std::string header = "\n>> 找到设备 " + id + " 的信息：";
            mylog_info(header.c_str());

            std::stringstream ss;
            ss << "   设备号: " << current->deviceId << "\n"
               << "   冷却时间: " << current->coolDownTime << "秒\n"
               << "   使用时间: " << current->usageTime << "秒";
            mylog_info(ss.str().c_str());
            return true;
        }
        current = current->next;
    }
    
    std::string msg = "未找到设备 " + id;
    mylog_warn(msg.c_str());
    return false;
}