#include "taskrunner.hpp"


TaskRunner::TaskRunner() : stopFlag(false) {
         try {
            worker = std::thread(&TaskRunner::run, this);
        }catch (...) {
            stopFlag = true;
            throw;
        }
}

TaskRunner::~TaskRunner() {
    {
        std::unique_lock<std::mutex> lock(mtx);
        stopFlag = true;
    }
    cv.notify_all();  // 唤醒线程准备退出
    if (worker.joinable()) {
        worker.join();
    }
}


void TaskRunner::addTask(const std::function<void()>& task) {
        {
            std::unique_lock<std::mutex> lock(mtx);
            tasks.push(task);
        }
        cv.notify_one();
}

void TaskRunner::run() {
    while (true) {
        std::function<void()> current_task; // 使用不会引起宏冲突的变量名

        {
            std::unique_lock<std::mutex> lock(mtx);
            cv.wait(lock, [this]() { return stopFlag || !tasks.empty(); });

            if (stopFlag && tasks.empty()) {
                break; // 退出线程
            }

            // 从队列中取出任务
            current_task = std::move(tasks.front());
            tasks.pop();
        }

        // --- 这是关键的修改部分 ---
        // 使用 if 语句来检查任务是否有效，并在有效时执行它。
        // 这一个代码块就替代了原来简单的 "task();"
        if (current_task) {
            current_task(); // 在 if 内部调用，确保 current_task 不是空的
        }
        // -------------------------
    }
}
