// mqtt_controller.hpp
#ifndef MQTT_CONTROLLER_HPP
#define MQTT_CONTROLLER_HPP

#include <string>
#include <functional>
#include <memory>
#include <atomic>
#include <mqtt/async_client.h> // 引入Paho C++库头文件

class MqttController {
public:
    // 定义消息回调函数的类型，方便用户使用
    using MessageCallback = std::function<void(const std::string& topic, const std::string& payload)>;

    // 构造函数：需要代理地址和客户端ID
    MqttController(const std::string& broker_address, const std::string& client_id);
    // 析构函数：自动断开连接
    ~MqttController();

    // 禁止拷贝和赋值，确保对象的唯一性
    MqttController(const MqttController&) = delete;
    MqttController& operator=(const MqttController&) = delete;

    // 连接到MQTT代理
    // @return 连接成功或正在连接返回true
    bool connect(const std::string& username, const std::string& password);

    // 断开连接
    void disconnect();

    // 检查客户端是否已连接
    bool isConnected() const;

    // 订阅一个主题，并设置一个专门处理该主题消息的回调函数
    void subscribe(const std::string& topic, MessageCallback callback);
    
    // 发布消息
    void publish(const std::string& topic, const std::string& payload, int qos = 1, bool retained = false);

private:
    // Paho库需要一个回调处理类，我们将其作为内部类以实现良好封装
    class CallbackHandler : public virtual mqtt::callback {
    public:
        CallbackHandler(MqttController& controller) : controller_(controller) {}

    private:
        // 当连接成功建立时调用
        void connected(const std::string& cause) override;
        // 当连接丢失时调用
        void connection_lost(const std::string& cause) override;
        // 当订阅的消息到达时调用
        void message_arrived(mqtt::const_message_ptr msg) override;
        
        MqttController& controller_;
    };

    // Paho异步客户端的智能指针
    std::unique_ptr<mqtt::async_client> client_;
    // 回调处理类的实例
    CallbackHandler callback_handler_;
    // 原子布尔值，用于线程安全地检查连接状态
    std::atomic<bool> is_connected_;

    // 存储用户为特定主题设置的回调函数
    MessageCallback message_callback_;
};

#endif // MQTT_CONTROLLER_HPP