#ifndef MQTT_CONTROLLER_HPP
#define MQTT_CONTROLLER_HPP

#include <string>
#include <vector>
#include <functional>
#include <memory>
#include <atomic>
#include <mutex>
#include <unordered_map>
#include <mqtt/async_client.h>

class MqttController {
public:
    // 定义回调函数类型
    using MessageCallback = std::function<void(const std::string& topic, const std::string& payload)>;
    using ConnectionSuccessCallback = std::function<void()>;
    using ConnectionLostCallback = std::function<void(const std::string& cause)>;

    // 构造函数：需要代理地址和客户端ID
    MqttController(const std::string& broker_address, const std::string& client_id);
    // 析构函数：自动断开连接
    ~MqttController();

    // 禁止拷贝和赋值，确保对象的唯一性
    MqttController(const MqttController&) = delete;
    MqttController& operator=(const MqttController&) = delete;

    // 连接到MQTT代理
    // @param username 用户名
    // @param password 密码
    // @param automatic_reconnect 是否启用自动重连
    // @return 连接请求已发送返回true，立即出错返回false
    bool connect(const std::string& username, const std::string& password, bool automatic_reconnect = true);

    // 断开连接
    void disconnect();

    // 检查客户端是否已连接
    bool isConnected() const;

    // 设置连接成功的回调
    void setOnConnectSuccessCallback(ConnectionSuccessCallback cb);
    // 设置连接丢失的回调
    void setOnConnectionLostCallback(ConnectionLostCallback cb);

    // 订阅一个主题
    void subscribe(const std::string& topic, int qos, MessageCallback callback);

    // 订阅多个主题（使用相同的回调和QoS）
    void subscribe(const std::vector<std::string>& topics, int qos, MessageCallback callback);

    // 取消订阅
    void unsubscribe(const std::string& topic);
    
    // 发布消息
    void publish(const std::string& topic, const std::string& payload, int qos = 1, bool retained = false);

private:
    // 内部类，用于处理来自Paho库的回调
    class CallbackHandler : public virtual mqtt::callback {
    public:
        CallbackHandler(MqttController& controller) : controller_(controller) {}

    private:
        void connected(const std::string& cause) override;
        void connection_lost(const std::string& cause) override;
        void message_arrived(mqtt::const_message_ptr msg) override;
        
        MqttController& controller_;
    };

    // 静态辅助函数，用于MQTT主题通配符匹配
    static bool topic_matches_sub(const std::string& sub, const std::string& topic);

    // Paho异步客户端的智能指针
    std::unique_ptr<mqtt::async_client> client_;
    // 回调处理类的实例
    CallbackHandler callback_handler_;
    // 原子布尔值，用于线程安全地检查连接状态
    std::atomic<bool> is_connected_;

    // 存储用户为特定主题设置的回调函数
    std::unordered_map<std::string, MessageCallback> topic_callbacks_;
    // 互斥锁，用于保护回调映射的线程安全访问
    std::mutex callbacks_mutex_;

    // 存储连接状态的回调
    ConnectionSuccessCallback on_connect_success_cb_;
    ConnectionLostCallback on_connection_lost_cb_;
};

#endif // MQTT_CONTROLLER_HPP