// serial_port.hpp
#ifndef SERIAL_PORT_HPP
#define SERIAL_PORT_HPP

#include <string>
#include <vector>
#include <functional>
#include <thread>
#include <atomic>
#include <mutex>

// POSIX/Linux headers
#include <termios.h>

class SerialPort {
public:
    // 定义消息处理回调函数的类型
    using MessageHandler = std::function<void(const std::vector<uint8_t>&)>;

    // 构造函数：传入串口设备名，如 "/dev/ttyUSB0"
    explicit SerialPort(const std::string& port_name);
    
    // 析构函数：自动关闭串口和监听线程
    ~SerialPort();

    // 禁止拷贝和赋值
    SerialPort(const SerialPort&) = delete;
    SerialPort& operator=(const SerialPort&) = delete;

    // 打开并配置串口
    // baud_rate: 如 B115200, B9600 等
    // returns: true on success, false on failure
    bool open(speed_t baud_rate = B115200);

    // 关闭串口
    void close();

    // 检查串口是否打开
    bool isOpen() const;

    // 发送数据（函数内部会自动计算并附加CRC16）
    // data: 要发送的原始数据
    // returns: true on success, false on failure
    bool send(const std::vector<uint8_t>& data);

    // 开始异步监听，当接收到完整消息时调用handler
    void startListening(MessageHandler handler);

    // 停止异步监听
    void stopListening();

    // 静态工具函数：CRC16校验计算
    static uint16_t crc16(const uint8_t* data, size_t len);

private:
    // 监听线程的主循环函数
    void listenerLoop();

    // 内部状态
    std::string port_name_;
    int fd_; // 文件描述符
    std::atomic<bool> is_open_;
    
    // 线程相关
    std::thread listener_thread_;
    std::atomic<bool> stop_listener_;
    MessageHandler message_handler_;
    std::mutex write_mutex_; // 保证发送操作的原子性
};

#endif // SERIAL_PORT_HPP