#include <sys/stat.h>
#include <dirent.h>
#include <stdarg.h>
#include "common.h"
#include "zlog.h"
#include "log.h"

const char *config_content = "[global]\n"
                             "strict init = true\n"
                             "buffer min = 1024\n"
                             "buffer max = 2MB\n"
                             "default format = \"%d(%Y-%m-%d %H:%M:%S.%ms) %-6V (%c:%F:%L) - %m%n\"\n"
                             "file perms = 600              # 设置文件权限\n"
                             "\n"
                             "[formats]\n"
                             "millisecond = \"%d(%Y-%m-%d %H:%M:%S.%ms) [%V] %m%n\"\n"
                             "\n"
                             "[rules]\n"
                             "# 日志文件路径，30天轮询，自动创建目录\n"
                             "my_log.*    \"/home/orangepi/car/master/log/log.log\", 30D; millisecond\n";

const char *LOG_FILE = "/home/orangepi/car/master/log/log.log";//日志文件

static zlog_category_t *zlog_c = NULL;

int logconf_create() {
    const char *config_file = "/home/orangepi/car/master/zlog.conf";
    // 检查文件是否存在
    if (access(config_file, F_OK) != 0) {
        // 文件不存在，创建并写入内容
        FILE *fp = fopen(config_file, "w");
        if (fp == NULL) {
            perror("Failed to create zlog.conf");
            return 1;
        }
        // 写入配置内容
        fwrite(config_content, strlen(config_content), 1, fp);
        fclose(fp);
        printf("zlog.conf has been created.\n");
    } else {
        printf("zlog.conf already exists.\n");
    }
    return 0;
}

// 检查并创建日志目录
static int create_log_dir_if_not_exist(const char *path) {
    char dir_path[256] = {0};
    const char *p = path;
    char *q = dir_path;
    
    // 提取目录部分
    while (*p != '\0') {
        *q = *p;
        if (*p == '/' && *(p+1) != '\0') {
            // 检查并创建每一级目录
            if (access(dir_path, F_OK) != 0) {
                if (mkdir(dir_path, 0755) != 0) {
                    fprintf(stderr, "Failed to create directory: %s\n", dir_path);
                    return -1;
                }
            }
        }
        p++;
        q++;
    }
    
    return 0;
}

int logprject_Init() {
    int rc;
    zlog_category_t *c;
    if(create_log_dir_if_not_exist(LOG_FILE)!=0) {
        fprintf(stderr, "Failed to create log directory\n");
        return -1;
    }

     // 初始化 zlog
    rc = zlog_init("/home/orangepi/car/master/zlog.conf");
    if (rc) {
        fprintf(stderr, "zlog init failed\n");
        return -2;
    }
    
    // 获取日志分类
    zlog_c = zlog_get_category("my_log");
    if (!c) {
        fprintf(stderr, "get category failed\n");
        zlog_fini();
        return -3;
    }
    
    return 0;
}


// 封装 zlog_debug
void my_zlog_debug(const char *format, ...) {
    if (zlog_c) {
        va_list args;
        va_start(args, format);
        vzlog_debug(zlog_c, format, args);
        va_end(args);
    }
}

// 封装 zlog_info
void my_zlog_info(const char *format, ...) {
    if (zlog_c) {
        va_list args;
        va_start(args, format);
        vzlog_info(zlog_c, format, args);
        va_end(args);
    }
}

// 封装 zlog_notice
void my_zlog_notice(const char *format, ...) {
    if (zlog_c) {
        va_list args;
        va_start(args, format);
        vzlog_notice(zlog_c, format, args);
        va_end(args);
    }
}

// 封装 zlog_warn
void my_zlog_warn(const char *format, ...) {
    if (zlog_c) {
        va_list args;
        va_start(args, format);
        vzlog_warn(zlog_c, format, args);
        va_end(args);
    }
}

// 封装 zlog_error
void my_zlog_error(const char *format, ...) {
    if (zlog_c) {
        va_list args;
        va_start(args, format);
        vzlog_error(zlog_c, format, args);
        va_end(args);
    }
}

// 封装 zlog_fatal
void my_zlog_fatal(const char *format, ...) {
    if (zlog_c) {
        va_list args;
        va_start(args, format);
        vzlog_fatal(zlog_c, format, args);
        va_end(args);
    }
}

// 结束 zlog
void my_log_close() {
    if (zlog_c) {
        zlog_fini();
        zlog_c = NULL;
    }
}
