#include "ads1115.h"
#include "common.h"
#include "hardware_iic.h"

#define ADS1115_ADDR 0x48 // ADS1115 的默认 I2C 地址

// I2C写寄存器
// reg: 寄存器地址
// val: 要写入的16位值
static void ads1115_write_register(uint8_t reg, uint16_t val) {
    uint8_t data[2];
    data[0] = (val >> 8) & 0xFF; // MSB
    data[1] = val & 0xFF;        // LSB

    if (hardware_iic_write(ADS1115_ADDR, reg, data, 2) != 0) {
        // 这里可以添加更详细的错误处理
        my_zlog_error("ADS1115: 写入寄存器 0x%02X 失败。", reg);
    }
}

// 读取转换寄存器
static uint16_t ads1115_read_conversion() {
    uint8_t data[2];
    if (hardware_iic_read(ADS1115_ADDR, 0x00, data, 2) != 0) {
        // 这里可以添加更详细的错误处理
        my_zlog_error("ADS1115: 读取转换寄存器失败。\n");
        return 0; // 返回0表示读取失败，可能需要更合适的错误值
    }
    return (data[0] << 8) | data[1];
}

// 读取某个单端通道电压，channel = 0~3
float ads1115_read_channel(int channel) {
    if (channel < 0 || channel > 3) {
        my_zlog_warn("ADS1115: 无效的通道号 %d (应为 0-3)。\n", channel);
        return -1.0f;
    }

    uint16_t config = 0x8000;       // OS=1，启动单次转换
    config |= ((4 + channel) << 12); // 单端MUX (AIN0: 0100, AIN1: 0101, AIN2: 0110, AIN3: 0111)
    config |= (0x00 << 9);          // PGA ±6.144V (FSR = 6.144V)
    config |= (1 << 8);              // 单次模式 (MODE = 1)
    config |= (0x04 << 5);          // 128SPS (DR = 100)
    config |= 0x03;                  // 禁用比较器 (COMP_MODE = 0, COMP_POL = 0, COMP_LAT = 0, COMP_QUE = 11)

    // 写入配置寄存器 (0x01)
    ads1115_write_register(0x01, config);

    // 等待转换完成。根据 128SPS，转换时间约为 1000ms / 128 = 7.8125ms。
    // 这里等待 10ms 是安全的。
    delay_ms(10);

    // 读取转换结果
    int16_t val = (int16_t)ads1115_read_conversion();

    // 将 16 位有符号值转换为电压
    // FSR = 6.144V, 2^15 = 32768
    float voltage = val * 6.144f / 32768.0f;

    // my_zlog_info("ADS1115: 通道 %d 原始值: %d, 电压: %.3f V", channel, val, voltage);

    return voltage;
}