#include "download.h"
#include "device_change.h"
#include "common.h"

// 用于存储下载数据的结构体
struct MemoryStruct {
    char *memory;
    size_t size;
};

// 回调函数，用于处理下载的数据
static size_t WriteMemoryCallback(void *contents, size_t size, size_t nmemb, void *userp) {
    size_t realsize = size * nmemb;
    struct MemoryStruct *mem = (struct MemoryStruct *)userp;
    
    char *ptr = realloc(mem->memory, mem->size + realsize + 1);
    if(!ptr) {
        my_zlog_warn("not enough memory (realloc returned NULL)");
        return 0;
    }
    
    mem->memory = ptr;
    memcpy(&(mem->memory[mem->size]), contents, realsize);
    mem->size += realsize;
    mem->memory[mem->size] = 0;
    
    return realsize;
}

int download_file(const char *url,const char *target_dir ,const char *filename) {
    // 创建目标目录（如果不存在）
    if (create_directory_if_not_exists(target_dir) == -1) {
        return 1;
    }

    // 构建完整文件路径
    char filepath[1024];
    snprintf(filepath, sizeof(filepath), "%s/%s", target_dir, filename);

    // 检查文件是否已存在
    if (file_exists(filepath)) {
        my_zlog_warn("文件已存在: %s\n跳过下载.\n", filepath);
        return 0;
    }

    CURL *curl_handle;
    CURLcode res;

    struct MemoryStruct chunk;
    chunk.memory = malloc(1);  // 初始化为空
    chunk.size = 0;

    curl_global_init(CURL_GLOBAL_ALL);
    curl_handle = curl_easy_init();

    if (curl_handle) {
        // 设置URL
        curl_easy_setopt(curl_handle, CURLOPT_URL, url);
        // 设置回调函数
        curl_easy_setopt(curl_handle, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);
        // 设置回调数据
        curl_easy_setopt(curl_handle, CURLOPT_WRITEDATA, (void *)&chunk);
        // 设置用户代理（有些网站需要）
        curl_easy_setopt(curl_handle, CURLOPT_USERAGENT, "libcurl-agent/1.0");
        // 跟随重定向
        curl_easy_setopt(curl_handle, CURLOPT_FOLLOWLOCATION, 1L);

        // 执行请求
        res = curl_easy_perform(curl_handle);

        // 检查错误
        if (res != CURLE_OK) {
            fprintf(stderr, "curl_easy_perform() 失败: %s\n", curl_easy_strerror(res));
        } else {
            // 将下载的数据写入文件
            FILE *fp = fopen(filepath, "wb");
            if (fp) {
                fwrite(chunk.memory, 1, chunk.size, fp);
                fclose(fp);
                my_zlog_debug("文件已下载到: %s\n", filepath);
            } else {
                fprintf(stderr, "无法打开文件 %s 进行写入\n", filepath);
            }
        }

        // 清理curl
        curl_easy_cleanup(curl_handle);
        free(chunk.memory);
    }

    curl_global_cleanup();

    return 0;
}