#ifndef PTHRPOLL_H
#define PTHRPOLL_H

// 为了让头文件自给自足，直接包含它所需要的依赖
#include "common.h"

// 任务结构体
typedef struct Task {
    void (*function)(void *);   // 函数指针，指向要执行的任务函数
    void *argument;             // 传递给任务函数的参数
    struct Task *next;          // 指向下一个任务的指针，构成链表
} Task;

// 任务队列结构体
typedef struct {
    Task *head;             // 队列头部
    Task *tail;             // 队列尾部
    int size;               // 队列中任务的数量
    pthread_mutex_t mutex;  // 用于保护任务队列的互斥锁
    pthread_cond_t cond;    // 用于线程同步的条件变量
} TaskQueue;

// 线程池结构体 (此结构体已被简化和明确化)
typedef struct {
    pthread_t *threads;       // 存放线程ID的数组
    int min_threads;          // 线程池中最小线程数
    int max_threads;          // 线程池中最大线程数
    
    // --- 字段已重命名和简化 ---
    int live_threads;         // 当前存活的线程数
    int busy_threads;         // 当前正在忙碌（执行任务）的线程数（为未来功能增强预留）

    TaskQueue task_queue;     // 任务队列
    int shutdown;             // 线程池关闭标志（1表示关闭，0表示运行）
    pthread_mutex_t mutex;    // 用于保护线程池级别变量（如线程计数）的互斥锁

    // --- 已移除的字段 ---
    // pthread_t reaper_thread;   // 已移除：回收线程的逻辑复杂且存在缺陷。
    // pthread_cond_t reaper_cond; // 已移除：没有回收线程后不再需要。
    // 'thread_count' 和 'active_threads' 的概念现在由 'live_threads' 更清晰地表示。

} ThreadPool_t;

/**
 * @brief 初始化线程池。
 * @param min_threads 线程池中保持存活的最小线程数。
 * @param max_threads 线程池可以扩展到的最大线程数。
 * @return 成功时返回新创建的线程池指针，失败时返回 NULL。
 */
ThreadPool_t *thread_pool_init(int min_threads, int max_threads);

/**
 * @brief 向线程池的任务队列中添加一个新任务。
 * @param pool 要添加任务的线程池。
 * @param function 要执行的任务函数指针。
 * @param argument 传递给任务函数的参数。
 *                 注意：调用者负责管理此参数的内存。
 *                 如果参数是动态分配的，任务函数内部应负责释放它。
 * @return 成功返回 0，失败返回 -1。
 */
int thread_pool_add_task(ThreadPool_t *pool, void (*function)(void *), void *argument);

/**
 * @brief 关闭并清理线程池资源。
 *        此函数会等待所有当前正在运行的任务执行完毕，但不会执行队列中剩余的任务。
 * @param pool 要销毁的线程池。
 */
void thread_pool_destroy(ThreadPool_t *pool);

#endif // PTHRPOLL_H